/*
 * Decompiled with CFR 0.152.
 */
package org.apache.commons.jcs3.auxiliary.disk.indexed;

import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.file.StandardOpenOption;
import org.apache.commons.jcs3.auxiliary.disk.indexed.IndexedDiskElementDescriptor;
import org.apache.commons.jcs3.engine.behavior.IElementSerializer;
import org.apache.commons.jcs3.log.Log;
import org.apache.commons.jcs3.log.LogManager;

public class IndexedDisk
implements AutoCloseable {
    public static final byte HEADER_SIZE_BYTES = 4;
    private final IElementSerializer elementSerializer;
    private static final Log log = LogManager.getLog(IndexedDisk.class);
    private final String filepath;
    private final FileChannel fc;

    public IndexedDisk(File file, IElementSerializer elementSerializer) throws IOException {
        this.filepath = file.getAbsolutePath();
        this.elementSerializer = elementSerializer;
        this.fc = FileChannel.open(file.toPath(), StandardOpenOption.CREATE, StandardOpenOption.READ, StandardOpenOption.WRITE);
    }

    protected <T> T readObject(IndexedDiskElementDescriptor ded) throws IOException, ClassNotFoundException {
        String message = null;
        boolean corrupted = false;
        long fileLength = this.fc.size();
        if (ded.pos > fileLength) {
            corrupted = true;
            message = "Record " + ded + " starts past EOF.";
        } else {
            ByteBuffer datalength = ByteBuffer.allocate(4);
            this.fc.read(datalength, ded.pos);
            datalength.flip();
            int datalen = datalength.getInt();
            if (ded.len != datalen) {
                corrupted = true;
                message = "Record " + ded + " does not match data length on disk (" + datalen + ")";
            } else if (ded.pos + (long)ded.len > fileLength) {
                corrupted = true;
                message = "Record " + ded + " exceeds file length.";
            }
        }
        if (corrupted) {
            log.warn("\n The file is corrupt: \n {0}", message);
            throw new IOException("The File Is Corrupt, need to reset");
        }
        ByteBuffer data = ByteBuffer.allocate(ded.len);
        this.fc.read(data, ded.pos + 4L);
        data.flip();
        return this.elementSerializer.deSerialize(data.array(), null);
    }

    protected void move(IndexedDiskElementDescriptor ded, long newPosition) throws IOException {
        int chunkSize;
        ByteBuffer datalength = ByteBuffer.allocate(4);
        this.fc.read(datalength, ded.pos);
        datalength.flip();
        int length = datalength.getInt();
        if (length != ded.len) {
            throw new IOException("Mismatched memory and disk length (" + length + ") for " + ded);
        }
        long readPos = ded.pos;
        long writePos = newPosition;
        ByteBuffer buffer = ByteBuffer.allocate(16384);
        for (int remaining = 4 + length; remaining > 0; remaining -= chunkSize) {
            chunkSize = Math.min(remaining, buffer.capacity());
            buffer.limit(chunkSize);
            this.fc.read(buffer, readPos);
            buffer.flip();
            this.fc.write(buffer, writePos);
            buffer.clear();
            writePos += (long)chunkSize;
            readPos += (long)chunkSize;
        }
        ded.pos = newPosition;
    }

    protected boolean write(IndexedDiskElementDescriptor ded, byte[] data) throws IOException {
        long pos = ded.pos;
        if (log.isTraceEnabled()) {
            log.trace("write> pos={0}", pos);
            log.trace("{0} -- data.length = {1}", this.fc, data.length);
        }
        if (data.length != ded.len) {
            throw new IOException("Mismatched descriptor and data lengths");
        }
        ByteBuffer headerBuffer = ByteBuffer.allocate(4);
        headerBuffer.putInt(data.length);
        headerBuffer.flip();
        int written = this.fc.write(headerBuffer, pos);
        assert (written == 4);
        ByteBuffer dataBuffer = ByteBuffer.wrap(data);
        written = this.fc.write(dataBuffer, pos + 4L);
        return written == data.length;
    }

    protected <T> void writeObject(T obj, long pos) throws IOException {
        byte[] data = this.elementSerializer.serialize(obj);
        this.write(new IndexedDiskElementDescriptor(pos, data.length), data);
    }

    protected long length() throws IOException {
        return this.fc.size();
    }

    @Override
    public void close() throws IOException {
        this.fc.close();
    }

    protected synchronized void reset() throws IOException {
        log.debug("Resetting Indexed File [{0}]", this.filepath);
        this.fc.truncate(0L);
        this.fc.force(true);
    }

    protected void truncate(long length) throws IOException {
        log.info("Truncating file [{0}] to {1}", this.filepath, length);
        this.fc.truncate(length);
    }

    protected String getFilePath() {
        return this.filepath;
    }
}

