/*
RSSFeedList.m

Author: Makoto Kinoshita

Copyright 2004-2006 The Shiira Project. All rights reserved.

Redistribution and use in source and binary forms, with or without modification, are permitted 
provided that the following conditions are met:

  1. Redistributions of source code must retain the above copyright notice, this list of conditions 
  and the following disclaimer.

  2. Redistributions in binary form must reproduce the above copyright notice, this list of 
  conditions and the following disclaimer in the documentation and/or other materials provided 
  with the distribution.

THIS SOFTWARE IS PROVIDED BY THE SHIIRA PROJECT ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, 
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE SHIIRA PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
POSSIBILITY OF SUCH DAMAGE.
*/

#import "RSSFeedList.h"
#import "RSSPersistentStack.h"

@interface RSSFeedList (private)
- (void)_updateTreeRSSFeeds;
@end

@implementation RSSFeedList

//--------------------------------------------------------------//
#pragma mark -- Initialize --
//--------------------------------------------------------------//

+ (id)sharedInstance
{
    static RSSFeedList* _sharedInstance = nil;
    if (!_sharedInstance) {
        _sharedInstance = [[RSSFeedList alloc] init];
    }
    
    return _sharedInstance;
}

- (id)init
{
    self = [super init];
    if (!self) {
        return nil;
    }
    
    // Initialize instance variables
    _treeRSSFeeds = [[NSMutableArray array] retain];
    
    // Register notification
    NSNotificationCenter*   center;
    center = [NSNotificationCenter defaultCenter];
    
    // Update tree RSS feeds
    [self _updateTreeRSSFeeds];
    
    return self;
}

- (void)dealloc
{
    // Release instance variables
    [_treeRSSFeeds release], _treeRSSFeeds = nil;
    
    // Remove observer
    [[NSNotificationCenter defaultCenter] removeObserver:self];
    
    [super dealloc];
}

//--------------------------------------------------------------//
#pragma mark -- RSS --
//--------------------------------------------------------------//

- (void)_updateTreeRSSFeeds
{
    // Remove old RSS feeds
    [_treeRSSFeeds removeAllObjects];
    
    // Get managed object context
    NSManagedObjectContext* context;
    context = [[RSSPersistentStack sharedInstance] managedObjectContext];
    
    // Get RSS feeds
    NSFetchRequest*     request;
    NSSortDescriptor*   lastBuildDateSort;
    NSSortDescriptor*   titleSort;
    NSArray*            feeds;
    request = [[NSFetchRequest alloc] init];
    [request autorelease];
    [request setEntity:
            [NSEntityDescription entityForName:@"RSSFeed" inManagedObjectContext:context]];
    
    lastBuildDateSort = [[NSSortDescriptor alloc] initWithKey:@"lastBuildDate" ascending:YES];
    [lastBuildDateSort autorelease];
    titleSort = [[NSSortDescriptor alloc] initWithKey:@"title" ascending:YES];
    [titleSort autorelease];
    [request setSortDescriptors:[NSArray arrayWithObjects:lastBuildDateSort, titleSort, nil]];
    
    feeds = [context executeFetchRequest:request error:NULL];
    
    // Add feeds
    NSEnumerator*       enumerator;
    NSManagedObject*    feed;
    enumerator = [feeds objectEnumerator];
    while (feed = [enumerator nextObject]) {
        NSMutableDictionary*    feedInfo;
        feedInfo = [NSMutableDictionary dictionary];
        
        id  value;
        
        // Set title
        value = [feed valueForKey:@"title"];
        if (!value) {
            continue;
        }
        [feedInfo setObject:value forKey:@"title"];
        
        // Set last build date
        value = [feed valueForKey:@"lastBuildDate"];
        if (value) {
            [feedInfo setObject:value forKey:@"lastBuildDate"];
        }
        
        // Set link
        value = [feed valueForKey:@"link"];
        if (value) {
            [feedInfo setObject:value forKey:@"link"];
        }
        
        // Set feed URL
        value = [feed valueForKey:@"feedURL"];
        if (value) {
            [feedInfo setObject:value forKey:@"feedURL"];
        }
        
        // Set items
        NSSet*  items;
        items = [feed valueForKey:@"items"];
        [feedInfo setObject:[NSMutableArray array] forKey:@"items"];
        
        // Add feed info
        [_treeRSSFeeds addObject:feedInfo];
    }
    
#if 0
    // Notify update
    [[NSNotificationCenter defaultCenter] 
            postNotificationName:SRWebHistoryUpdated object:self];
#endif
}

- (NSArray*)treeRSSFeeds
{
    return _treeRSSFeeds;
}

@end
