/*
SRURLCompleteDataSource.m

Author: Makoto Kinoshita

Copyright 2004-2006 The Shiira Project. All rights reserved.

Redistribution and use in source and binary forms, with or without modification, are permitted 
provided that the following conditions are met:

  1. Redistributions of source code must retain the above copyright notice, this list of conditions 
  and the following disclaimer.

  2. Redistributions in binary form must reproduce the above copyright notice, this list of 
  conditions and the following disclaimer in the documentation and/or other materials provided 
  with the distribution.

THIS SOFTWARE IS PROVIDED BY THE SHIIRA PROJECT ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, 
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE SHIIRA PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
POSSIBILITY OF SUCH DAMAGE.
*/

#import "SRURLCompleteDataSource.h"

@interface SRURLCompleteDataSource (private)
+ (void)_addURLString:(NSString*)URLString append:(BOOL)flag;
+ (NSMutableArray*)_historyURLStrings;
@end

@implementation SRURLCompleteDataSource

//--------------------------------------------------------------//
#pragma mark -- Initialize --
//--------------------------------------------------------------//

- (id)init
{
    self = [super init];
    if (!self) {
        return nil;
    }
    
    // Initialize instance variables
    _URLStrings = [[NSMutableArray array] retain];
    _completedURLStringDict = [[NSMutableDictionary dictionary] retain];
    
    // Register notification
    static BOOL _isRegistered = NO;
    if (!_isRegistered) {
        NSNotificationCenter*   center;
        center = [NSNotificationCenter defaultCenter];
        
        [center addObserver:[self class] selector:@selector(webHistoryItemsAdded:) 
                name:WebHistoryItemsAddedNotification object:nil];
        [center addObserver:[self class] selector:@selector(webHistoryItemsRemoved:) 
                name:WebHistoryItemsRemovedNotification object:nil];
        [center addObserver:[self class] selector:@selector(webHistoryAllItemsRemoved:) 
                name:WebHistoryAllItemsRemovedNotification object:nil];
        
        _isRegistered = YES;
    }
    
    return self;
}

- (void)dealloc
{
    [_URLStrings release], _URLStrings = nil;
    [_completedURLStringDict release], _completedURLStringDict = nil;
    
    [super dealloc];
}

//--------------------------------------------------------------//
#pragma mark -- URL string copying --
//--------------------------------------------------------------//

+ (void)_addURLString:(NSString*)URLString append:(BOOL)flag
{
    NSMutableArray* historyURLStrings;
    historyURLStrings = [self _historyURLStrings];
    
    // Add domain name
    if ([URLString hasPrefix:@"http://"]) {
        NSString*   hostString = nil;
        NSRange     slashRange;
        slashRange = [URLString rangeOfString:@"/" 
                options:NSLiteralSearch range:NSMakeRange(7, [URLString length] - 7)];
        if (slashRange.length) {
            hostString = [URLString substringToIndex:slashRange.location];
        }
        
        if (hostString && ![historyURLStrings containsObject:hostString]) {
            if (flag) {
                [historyURLStrings addObject:hostString];
            }
            else {
                [historyURLStrings insertObject:hostString atIndex:0];
            }
        }
    }
    
    // Add URL string
    if (![historyURLStrings containsObject:URLString]) {
        if (flag) {
            [historyURLStrings addObject:URLString];
        }
        else {
            [historyURLStrings insertObject:URLString atIndex:0];
        }
    }
}

+ (NSMutableArray*)_historyURLStrings
{
    static NSMutableArray*  _historyURLStrings = nil;
    
    if (!_historyURLStrings) {
        _historyURLStrings = [[NSMutableArray array] retain];
        
        // Get shared history
        WebHistory* history;
        history = [WebHistory optionalSharedHistory];
        
        // Copy history URLs
        NSArray*        days;
        NSEnumerator*   dayEnumerator;
        NSCalendarDate* day;
        days = [history orderedLastVisitedDays];
        dayEnumerator = [days objectEnumerator];
        
        while (day = [dayEnumerator nextObject]) {
            NSArray*        items;
            NSEnumerator*   itemEnumerator;
            WebHistoryItem* item;
            items = [history orderedItemsLastVisitedOnDay:day];
            itemEnumerator = [items objectEnumerator];
            
            while (item = [itemEnumerator nextObject]) {
                [[self class] _addURLString:[item URLString] append:YES];
            }
        }
    }
    
    return _historyURLStrings;
}

//--------------------------------------------------------------//
#pragma mark -- Complete URL string with uncompleted string --
//--------------------------------------------------------------//

- (NSString*)absoluteURLStringOfCompletedString:(NSString*)uncompletedString
{
    // Sanitize string
    // Remove new line characters
    NSCharacterSet* newLineCharacterSet;
    newLineCharacterSet = [NSCharacterSet newLineCharacterSet];
    if ([uncompletedString rangeOfCharacterFromSet:newLineCharacterSet 
            options:NSLiteralSearch].location != NSNotFound)
    {
        uncompletedString = [uncompletedString stringByTrimmingCharactersInSet:newLineCharacterSet];
    }
    
#if 1
    // Remove last white spaces
    int length;
    for (length = [uncompletedString length]; length > 0; length--) {
        unichar c;
        c = [uncompletedString characterAtIndex:length - 1];
        if (c != 0x0020 && c != 0x0009) { // space and tab
            break;
        }
    }
    if (length < [uncompletedString length]) {
        uncompletedString = [uncompletedString substringWithRange:NSMakeRange(0, length)];
    }
#else
    uncompletedString = [uncompletedString stringByAddingPercentEscapesUsingEncoding:NSASCIIStringEncoding];
#endif
    
    // For empty string
    if ([uncompletedString length] == 0) {
        return nil;
    }
    
    // If passed string is well-formed, use it
    NSURL*  url;
    url = [NSURL _web_URLWithUserTypedString:uncompletedString];
    if (url && [url scheme]) {
        return uncompletedString;
    }
    
    // Get URL string from dictioanry
    NSString*   URLString;
    URLString = [_completedURLStringDict objectForKey:uncompletedString];
    if (!URLString) {
        NSString*   firstComponent = [[uncompletedString pathComponents] objectAtIndex:0];
        
        if ([firstComponent rangeOfString:@"."].length) {
            URLString = [NSString stringWithFormat:@"http://%@", uncompletedString];
        }
        else {
            NSRange pathRange = NSMakeRange([firstComponent length],
                                            [uncompletedString length] - [firstComponent length]);
            NSString*   relativePath = [uncompletedString substringWithRange:pathRange];
            URLString = [NSString stringWithFormat:@"http://www.%@.com%@", firstComponent, relativePath];
        }
    }
    
    return URLString;
}

//--------------------------------------------------------------//
#pragma mark -- NSURLComboBox data source --
//--------------------------------------------------------------//

- (int)numberOfItemsInComboBox:(NSComboBox*)comboBox
{
    return [_URLStrings count];
}

- (id)comboBox:(NSComboBox*)comboBox objectValueForItemAtIndex:(int)index
{
    if (index < 0 || index >= [_URLStrings count]) {
        return [comboBox stringValue];
    }
    return [_URLStrings objectAtIndex:index];
}

- (NSString*)comboBox:(NSComboBox*)comboBox completedString:(NSString*)uncompletedString
{
    // Remove all objects in completed list
    [_URLStrings removeAllObjects];
    [_completedURLStringDict removeAllObjects];
    
    // Check all URLs
    NSMutableDictionary*    completedURLStringDict;
    NSEnumerator*           enumerator;
    NSString*               URLString;
    completedURLStringDict = [NSMutableDictionary dictionary];
    enumerator = [[SRURLCompleteDataSource _historyURLStrings] objectEnumerator];
    while (URLString = [enumerator nextObject]) {
        
        // Case of uncomplete string starts with 'http://'
        if ([uncompletedString hasPrefix:@"http://"]) {
            if ([URLString hasPrefix:uncompletedString]) {
                [_URLStrings addObject:URLString];
                [_completedURLStringDict setObject:URLString forKey:uncompletedString];
                [completedURLStringDict setObject:uncompletedString forKey:URLString];
            }
        }
        else {
            // Make URL string without 'http://'
            if ([URLString hasPrefix:@"http://"]) {
                NSString*   URLStringWithoutHttp;
                URLStringWithoutHttp = [URLString substringFromIndex:7];
                if ([URLStringWithoutHttp hasPrefix:uncompletedString]) {
                    [_URLStrings addObject:URLString];
                    [_completedURLStringDict setObject:URLString forKey:URLStringWithoutHttp];
                    [completedURLStringDict setObject:URLStringWithoutHttp forKey:URLString];
                }
                // Make URL string without 'http://www.'
                else if ([URLString hasPrefix:@"http://www."]) {
                    NSString*   URLStringWithoutWWW;
                    URLStringWithoutWWW = [URLString substringFromIndex:11];
                    if ([URLStringWithoutWWW hasPrefix:uncompletedString]) {
                        [_URLStrings addObject:URLString];
                        [_completedURLStringDict setObject:URLString forKey:URLStringWithoutWWW];
                        [completedURLStringDict setObject:URLStringWithoutWWW forKey:URLString];
                    }
                }
            }
        }
    }
    
    // Sort array
    [_URLStrings sortUsingSelector:@selector(compare:)];
    
    // Notify the change of number of items
    [comboBox noteNumberOfItemsChanged];
    
    // Check number of completed URLs
    if ([_URLStrings count] == 0) {
        return uncompletedString;
    }
    
    return [completedURLStringDict objectForKey:[_URLStrings objectAtIndex:0]];
}

//--------------------------------------------------------------//
#pragma mark -- WebHistory notification --
//--------------------------------------------------------------//

+ (void)webHistoryItemsAdded:(NSNotification*)notification
{
    // Get items
    NSArray*    items;
    items = [[notification userInfo] objectForKey:WebHistoryItemsKey];
    
    // Add items
    NSEnumerator*   enumerator;
    WebHistoryItem* item;
    enumerator = [items objectEnumerator];
    while (item = [enumerator nextObject]) {
        // Get URL string
        NSString*   URLString;
        URLString = [item URLString];
        
        [self _addURLString:URLString append:NO];
    }
}

+ (void)webHistoryItemsRemoved:(NSNotification*)notification
{
    // Get items
    NSArray*    items;
    items = [[notification userInfo] objectForKey:WebHistoryItemsKey];
    
    // Remove items
    NSEnumerator*   enumerator;
    WebHistoryItem* item;
    enumerator = [items objectEnumerator];
    while (item = [enumerator nextObject]) {
        // Get URL string
        NSString*   URLString;
        URLString = [item URLString];
        
        [[self _historyURLStrings] removeObject:URLString];
    }
}

+ (void)webHistoryAllItemsRemoved:(NSNotification*)notification
{
    // Remove all objects
    [[self _historyURLStrings] removeAllObjects];
}

@end
