﻿// Copyright (c) Microsoft Corporation 2005-2007.
// This sample code is provided "as is" without warranty of any kind. 
// We disclaim all warranties, either express or implied, including the 
// warranties of merchantability and fitness for a particular purpose. 
//

#light

#r "Northwind.dll"


open System
open System.Data
open System.Data.Linq
open System.Collections
open System.Collections.Generic
open nwind
open System.IO
open System.Windows.Forms
open System.Data.SqlClient
open System.Transactions
open Microsoft.FSharp.Quotations.Typed
open Microsoft.FSharp.Linq
open Microsoft.FSharp.Linq.SequenceOps
open Microsoft.FSharp.Data.Linq
open Sample.Support
open Sample.NorthwindDumper

[<ReflectedDefinition>]
let (|>) x f = f x

let dbPath() = 
    let dbPath1 = Path.GetFullPath(Path.Combine(Application.StartupPath, @"NORTHWND.MDF"))
    let dbPath2 = Path.GetFullPath(Path.Combine(System.Environment.GetFolderPath(System.Environment.SpecialFolder.MyDocuments), @"NORTHWND.MDF"))
    if   System.IO.File.Exists dbPath1 then dbPath1
    elif System.IO.File.Exists dbPath2 then dbPath2
    else 
       System.Windows.Forms.MessageBox.Show("You must install a version of Microsoft SQLServer or SQLServer Express and download the Northwind example database (NORTHWND.MDF) before running the database portions of this sample. You can download NORTHWND.MDF from http://www.codeproject.com/cs/database/InstallingNorthwindAndPub.asp. Place it in either "+dbPath1+" or the FLinq sample directory or your My Documents directory. Then restart this sample.") |> ignore; 
       dbPath1

let sqlServerInstance = @".\SQLEXPRESS"
let connString = "AttachDBFileName='" + dbPath() + "';Server='" + sqlServerInstance + "';user instance=true;Integrated Security=SSPI;Connection Timeout=30"

//let connString = "Server=.; Database=Northwind; Integrated Security=SSPI";

// This is mutable because we reset it after running each sample
let mutable db = new Northwind(connString)
onEachSample (fun () -> db.Log <- Console.Out)

let clearDBCache() = 
    let oldLog = db.Log 
    db <- new Northwind(connString);
    db.Log <- oldLog

//-------------------------------------------------------------------------
// The samples

[<Category("BASICS");
  Title("Dumping a table");
  Description("This sample dumps the Customers table")>]
let DLinq0a() =
    SQL <@ %db.Customers @>
    |> dump








//-------------------------------------------------------------------------
// Sample

[<Category("BASICS");
  Title("Select - Identity function");
  Description("This sample uses select with an identity function to select all the rows of the Customers table. \
               The F# syntax for an anonymous lambda expression is 'fun x -> y'.")>]
let DLinq0b() =
    SQL <@ seq { for c in %db.Customers -> c } @>
    |> dump 









//-------------------------------------------------------------------------
// Sample

[<Category("BASICS");
  Title("Select - Simple 0a");
  Description("This sample uses select to project out the City \
               column from the Customers table.  Notice how no type annotation is \
               needed on the 'c' variable: F#'s type inference has propagated the \
               type information from the customers table through the quoted expression.")>]
let DLinq0c() =
    SQL <@ seq { for c in %db.Customers -> c.City } @>
    |> dump







//----------------------------------------------
[<Support("DLinq1c");
 ReflectedDefinition>]
let cityIs (c: Customer) city = (c.City = city)

[<Category("WHERE");
  Title("Where - Using a top level definition");
  Description("This sample uses a top level definition as part of a query.  This definition \
               is available for use both regular F# code and quoted expressions. The FLINQ \
               layer rewrites out public, closed top-level definitions as macros. Here \
               'public' means 'revealed a module's signature' and 'closed' means \
               'has a definition that does not refer to any top level definitions hidden \
               by a module.'  Reflected top level definitions are only available for assemblies \
               compiled using the --quotation-data flag and the lambda term for the \
               definitions is attached as an attribute to the compiled assembly.  \
               The LINQ library itself provides way to utilize top-level \
               definitions as part of System.Expressions.Expression specifications, \
               except as lifted 'function objects' which can only be run, and not \
               transformed to SQL.")>]

let DLinq1c() =
    SQL <@ seq { for c in %db.Customers
                 when cityIs c "London"
                 -> c } @>
    |> dump 



              
//System.Environment.GetFolderPath
//System.Environment.SpecialFolder







//----------------------------------------------
[<Category("WHERE");
  Title("Where - Simple 3");
  Description("This sample uses where to filter for Products that have stock below their \
                     reorder level and are not discontinued.")>]
let DLinq3() =
    SQL <@ seq { for p in %db.Products
                 when p.UnitsInStock <= p.ReorderLevel && not p.Discontinued 
                 -> p } @>

    |> dump

              








//----------------------------------------------
[<Category("WHERE");
  Title("First - Simple");
  Description("This sample uses First to select a single Customer.")>]
let DLinq5() =
    SQL <@ seq { for c in %db.Customers -> c }
           |> first (fun c -> c.City = "Berlin" ) @>
    |> Seq.singleton
    |> dump 
        

              








//----------------------------------------------
[<Category("DISTINCT");
  Title("Distinct - Simple");
  Description("This sample uses Distinct to select the list of unique cities \
                     that have Customers.")>]
let DLinq6() =
    SQL <@ seq { for c in %db.Customers -> c.City }
           |> distinct @>
    |> dump

              








//----------------------------------------------
[<Category("SELECT");
  Title("Select - Simple");
  Description("This sample uses select to return a sequence of just the \
               Customers' contact names.")>]
let DLinq7() =
    SQL <@ seq { for c in %db.Customers -> c.ContactName } @>
    |> dump

              








//----------------------------------------------
[<Category("SELECT");
  Title("Select - Anonymous Type 1");
   Description("This sample uses select and anonymous types to return \
               a sequence of just the Customers' contact names and phone numbers.")>]
let DLinq8() =
    SQL <@ seq { for c in %db.Customers -> (c.ContactName, c.Phone) } @>
    |> dump
        
              








//----------------------------------------------
[<Category("SELECT");
  Title("Select - Anonymous Type 2");
  Description("This sample uses select and to return \
               a sequence of just the Categories' names and list of Products.")>]
let DLinq9() =
    SQL <@ seq { for e in %db.Employees ->
                   e.FirstName + " " + e.LastName, e.HomePhone } @>
    |> dump
              








//----------------------------------------------
[<Category("SELECT/DISTINCT");
  Title("Select - Filtered");
  Description("This sample uses select and where to return a sequence of \
               just the London Customers' contact names.")>]
let DLinq11() =
    SQL <@ seq { for c in %db.Customers
                 when c.City = "London"
                 -> c.ContactName } @>
    |> dump


              








//----------------------------------------------
[<Category("SELECT/DISTINCT");
  Title("Select - Shaped");
  Description("This sample uses select and anonymous types to return \
               a shaped subset of the data about Customers.")>]
let DLinq12() =
    SQL <@ seq { for c in %db.Customers
                 -> (c.CustomerID,
                           (c.CompanyName, c.City, c.Country),
                           (c.ContactName, c.ContactTitle)) } @>
    |> dump



              








//----------------------------------------------
[<Category("GROUPBY");
  Title("GroupBy - Simple grouping");
  Description("This sample uses groupBy to partition a table.")>]
let DLinq21a() =
    SQL <@ %db.Products
           |> groupBy (fun p -> p.CategoryID) @>
    |> dump 

              








//----------------------------------------------
[<Category("GROUPBY");
  Title("GroupBy - Selecting from groups");
  Description("This sample uses groupBy/select to partition a table and select the key/group from each group.")>]
let DLinq21b() =
    SQL <@ seq { for g in %db.Products |> groupBy (fun p -> p.CategoryID) 
                 -> (g.Key,g) } @>
    |> dump 

              








//----------------------------------------------
#if NOT_WORKING
[<Category("GROUPBY");
  Title("GroupBy - Multiple Columns");
  Description("This sample uses Group By to group products by CategoryID and SupplierID.")>]
let DLinq21c() =
    SQL <@ seq { for g in %db.Products |> groupBy (fun p -> p.CategoryID, p.SupplierID)
                 -> (g.Key,g) } @>
    |> dump 
#endif

              








//----------------------------------------------
 
[<Category("GROUPBY");
  Title("GroupBy - Partition");
  Description("")>]

let DLinq21g() =
    SQL <@ seq { for g in %db.Products |> groupBy (fun p -> p.CategoryID) 
                 let lowest = g |> fmin (fun p3 -> p3.UnitPrice)
                 let group = { for p in g when p.UnitPrice = lowest -> p }
                 -> (g.Key,group) } @>

    |> dump 

        






//----------------------------------------------
[<Category("JOIN");
  Title("SelectMany - 1 to Many - 1");
  Description("This sample uses foreign key navigation in the from clause to select all orders for customers in London")>]
let DLinqJoin1() =
    SQL <@ seq { for c in %db.Customers 
                 for d in c.Orders 
                 when c.City = "London" 
                 -> d } @>
    |> dump



//----------------------------------------------
[<Category("JOIN");
  Title("Join and Filter Two Independent Tables");
  Description("This sample takes a filtered product of two tables")>]
let DLinqJoin1a() =
    SQL <@ seq { for c in %db.Customers
                 for d in %db.Orders
                 when c.City = "London" 
                 -> (c,d) } @>
    |> dump













//----------------------------------------------
[<Category("JOIN");
  Title("SelectMany - 1 to Many - 2");
  Description("This sample uses foreign key navigation in the where clause to filter for Products whose Supplier is in the USA that are out of stock.This sample uses foreign key navigation in the from clause to select all orders for customers in London")>]
let DLinqJoin2() =
    SQL <@ seq { for p in %db.Products
                 when p.Supplier.Country = "USA" && p.UnitsInStock =?! 0s
                 -> p } @>
    |> dump













//----------------------------------------------
[<Category("JOIN");
  Title("SelectMany - Many to Many");
  Description("This sample uses foreign key navigation the \
               from clause to filter for employees Seattle, \
               and also list their territories.")>]
let DLinqJoin3() =
    SQL <@ seq { for e in %db.Employees
                 for et in e.EmployeeTerritories 
                 when e.City = "Seattle" 
                 -> (e.FirstName, e.LastName, et.Territory.TerritoryDescription) } @>
    |> dump



              








//----------------------------------------------
[<Category("JOIN");
  Title("SelectMany - Self-Join");
 Description("This sample uses foreign key navigation the \
               select clause to filter for pairs of employees where \
               one employee reports to the other and where \
               both employees are from the same City.")>]
let DLinqJoin4() =
    SQL <@ seq { for e1 in %db.Employees
                 for e2 in e1.Employees 
                 when e1.City = e2.City
                 -> (e1.FirstName, e1.LastName, e2.FirstName, e2.LastName, e1.City) } @>
    |> dump
        

              








//----------------------------------------------
[<Category("JOIN");
  Title("SelectMany - Two way join");
 Description("This sample explictly joins two tables and projects results from both tables.")>]
let DLinqJoin5() =
    SQL <@ groupJoin (%db.Customers)
                     (%db.Orders)
                     // keys to join on
                     (fun c -> c.CustomerID) (fun o -> o.CustomerID) 
                     // data to collect
                     (fun c orders -> (c.ContactName, count orders))  @>
    |> dump
        

              








#if GROUPJOIN
//----------------------------------------------
[<Category("JOIN");
  Title("GroupJoin - Three way join");
 Description("This sample explictly joins three tables and projects results from each of them.")>]
let DLinqJoin6() =
    SQL <@ groupJoin 
               (groupJoin 
                   (%db.Customers)
                   (%db.Orders)
                   (fun c -> c.CustomerID) (fun o -> o.CustomerID) 
                   (fun c ords -> (c,ords)))
               (%db.Employees)
               (fun (c,ords) -> c.City)
               (fun e -> e.City)
               (fun (c,ords) emps -> (c.ContactName, count ords, count emps))  @>
    |> dump
              


//----------------------------------------------
[<Category("JOIN");
  Title("GroupJoin - LEFT OUTER JOIN");
 Description("This sample shows how to get LEFT OUTER JOIN by using DefaultIfEmpty(). The DefaultIfEmpty() method returns null when there is no Order for the Employee.")>]
let DLinqJoin7() =
    SQL <@ seq { for (e,ords) in 
                     groupJoin 
                         (%db.Employees)
                         (%db.Orders)
                         (fun e -> e) (fun o -> o.Employee) 
                         (fun e ords -> (e,ords))
                 for o in defaultIfEmpty ords
                 -> (e.FirstName, e.LastName, o) }  @>
    |> dump





//----------------------------------------------
[<Category("JOIN");
  Title("GroupJoin - Projected let assignment");
 Description("This sample projects a 'let' expression resulting from a join.")>]
let DLinqJoin8() =
    SQL <@ seq { for (c,ords) in 
                     groupJoin 
                         (%db.Customers)
                         (%db.Orders)
                         (fun c -> c.CustomerID) (fun o -> o.CustomerID) 
                         (fun c ords -> (c,ords))
                 let z = c.City + c.Country
                 for o in ords
                 -> (c.ContactName, o.OrderID, z) }  @>
    |> dump





//----------------------------------------------
[<Category("JOIN");
  Title("GroupJoin - Composite Key");
 Description("This sample shows a join with a composite key.")>]
let DLinqJoin9() =
    SQL <@ seq { for o in %db.Orders
                 for (p,details) in 
                     groupJoin 
                         (%db.Products)
                         (%db.OrderDetails)
                         (fun p -> (o.OrderID, p.ProductID)) (fun d -> (d.OrderID, d.ProductID)) 
                         (fun p details -> (p,details))
                 for d in details
                 -> (o.OrderID, p.ProductID, d.UnitPrice) }  @>
    |> dump

#endif // GROUPJOIN






//----------------------------------------------
[<Category("Paging");
  Title("Paging - Ordered Unique Key");
 Description("This sample uses a where clause and the Take operator to do paging by, \
               first filtering to get only the ProductIDs above 50 (the last ProductID \
               from page 5), then ordering by ProductID, and finally taking the first 10 results, \
               thereby providing the data for page 6 of the Products table.  \
               Note that this method only works when ordering by a unique key.")>]
let DLinq55() =
    SQL <@ seq { for p in %db.Products
                 when p.ProductID > 50 
                 -> p }
           |> orderBy (fun p -> p.ProductID)
           |> take 10 @>
    |> dump

              








//----------------------------------------------
[<Category("Simultaneous Changes");
  Title("Optimistic Concurrency - 1");
 Description("This and the following sample demonstrate optimistic concurrency.  In this sample, \
               the other user makes and commits his update to Product 1 before you read the data \
               so no conflict occurs.")>]
let DLinq63() =
    Console.WriteLine("OTHER USER: ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~");
                  
    // Open a second connection to the database to simulate another user
    // who is going to make changes to the Products table                
    let otherUser_db = new Northwind(connString) 
    otherUser_db.Log <- db.Log;
    let otherUser_product = SQL <@ %otherUser_db.Products |> first  (fun p -> p.ProductID = 1) @> 
    otherUser_product.UnitPrice <- new Nullable<_>(new Decimal(999.99));
    otherUser_db.SubmitChanges();

    Console.WriteLine("~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~");


    Console.WriteLine();
    Console.WriteLine("YOU:  ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~");
              
    let product = SQL <@ %db.Products |> first  (fun p -> p.ProductID = 1) @> 
    product.UnitPrice <- new Nullable<_>(new Decimal(777.77));
                  
    let conflict = 
        try 
            db.SubmitChanges(); false 
        with 
            | :? ChangeConflictException -> true 

    Console.WriteLine();
    if (conflict) then begin
      Console.WriteLine("* * * OPTIMISTIC CONCURRENCY EXCEPTION * * *");
      Console.WriteLine("Another user has changed Product 1 since it was first requested.");
      Console.WriteLine("Backing out changes.");
    end else begin
      Console.WriteLine("* * * COMMIT SUCCESSFUL * * *");
      Console.WriteLine("Changes to Product 1 saved.");
    end;

    Console.WriteLine("~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~")



              








//----------------------------------------------
[<Category("Simultaneous Changes");
  Title("Optimistic Concurrency - 2");
 Description("This and the previous sample demonstrate optimistic concurrency.  In this sample, \
               the other user makes and commits his update to Product 1 after you read the data, \
               but before completing your update, causing an optimistic concurrency conflict.  \
               Your changes are rolled back, allowing you to retrieve the newly updated data \
               from the database and decide how to proceed with your own update.")>]
let DLinq64() =
    Console.WriteLine("YOU:  ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~");
              
    let product = SQL <@ %db.Products |> first  (fun p -> p.ProductID = 1) @> 
              
    Console.WriteLine("~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~");

                  
    Console.WriteLine();
    Console.WriteLine("OTHER USER: ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~");
                  
    // Open a second connection to the database to simulate another user
    // who is going to make changes to the Products table
    let otherUser_db = new Northwind(connString) 
    otherUser_db.Log <- db.Log;
    let otherUser_product = SQL <@ %otherUser_db.Products |> first  (fun p -> p.ProductID = 1) @> 
    otherUser_product.UnitPrice <- new Nullable<_>(otherUser_product.UnitPrice.Value / new Decimal(2));
    otherUser_db.SubmitChanges();

    dump1 (SQL <@ %otherUser_db.Products |> first  (fun p -> p.ProductID = 1) @> );
    
    Console.WriteLine("~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~");


    Console.WriteLine();
    Console.WriteLine("YOU (continued): ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~");

    product.UnitPrice <- new Nullable<_>(product.UnitPrice.Value * new Decimal(2));
                  
    let conflict = 
       try 
           db.SubmitChanges(); false 
       with 
           | :? ChangeConflictException -> true 

    if (conflict) then begin
        Console.WriteLine("* * * OPTIMISTIC CONCURRENCY EXCEPTION * * *");
        Console.WriteLine("Another user has changed Product 1 since it was first requested.");
        Console.WriteLine("Backing out changes.");
    end else begin
        Console.WriteLine("* * * COMMIT SUCCESSFUL * * *");
        Console.WriteLine("Changes to Product 1 saved.");
    end;
    dump1 (SQL <@ %otherUser_db.Products |> first  (fun p -> p.ProductID = 1) @> );
    dump1 (SQL <@ %db.Products |> first  (fun p -> p.ProductID = 1) @>);

                








//----------------------------------------------
[<Category("Simultaneous Changes");
  Title("Transactions - Explicit");
 Description("This sample demonstrates open an explicit transaction.  This \
               provides more protection by including the reading of the data the \
               transaction to help prevent optimistic concurrency exceptions.  \
               As the previous query, the update to prod2's UnitsInStock field \
               makes the value negative, which violates a check constraint within \
               the database.  This causes the transaction that is updating both \
               Products to fail, which rolls back all changes.")>]
let DLinq66() =
    Console.WriteLine("*** BEFORE ***");
    dump (SQL <@ seq { for p in %db.Products when p.ProductID = 4 -> p } @>);
    dump (SQL <@ seq { for p in %db.Products when p.ProductID = 5 -> p } @>);
    Console.WriteLine();
    Console.WriteLine("*** UPDATE WITH EXPLICIT TRANSACTION ***");
    // Explicit use of TransactionScope ensures that
    // the data will not change the database between
    // read and write
    use ts = new TransactionScope()
    (   try 
            let prod1 = SQL <@ %db.Products |> first  (fun p -> p.ProductID = 4) @> 
            let prod2 = SQL <@ %db.Products |> first  (fun p -> p.ProductID = 5) @> 
            assert(prod1.UnitsInStock.HasValue);
            assert(prod2.UnitsInStock.HasValue);
            prod1.UnitsInStock <- new Nullable<_>(prod1.UnitsInStock.Value - 3s);
            prod2.UnitsInStock <- new Nullable<_>(prod2.UnitsInStock.Value - 5s); // ERROR: this will make the units stock negative
            db.SubmitChanges()
        with e -> Console.WriteLine(e.Message)
    );
    Console.WriteLine();
    Console.WriteLine("*** AFTER ***");
    clearDBCache();
    dump (SQL <@ seq { for p in %db.Products when p.ProductID = 4 -> p } @>);
    dump (SQL <@ seq { for p in %db.Products when p.ProductID = 5 -> p } @>);
            

              








//----------------------------------------------
[<Category("Object Loading");
  Title("Deferred Loading - 2");
 Description("This sample demonstrates how navigating through relationships \
               retrieved objects can end up triggering new queries to the database \
               if the data was not requested by the original query.")>]
let DLinq92() =
    let custs = SQL <@ seq { for c in %db.Customers when c.City = "London" -> c } @> 

    custs |> iter (fun cust -> 
        cust.Orders |> iter (fun ord -> 
            ord.OrderDetails |> iter (fun orderDetail -> 
              Console.WriteLine("CustomerID {0} has an OrderID {1} with ProductID {2} that has name {3}.",
                                  cust.CustomerID, ord.OrderID, orderDetail.ProductID, orderDetail.Product.ProductName)
            )
        )
    )

