(*F#
module Microsoft.FSharp.Compiler.PostTypecheckSemanticChecks
open Microsoft.Research.AbstractIL
open Microsoft.Research.AbstractIL.Internal
open Microsoft.FSharp.Compiler

module Il = Microsoft.Research.AbstractIL.IL
module Ildiag = Microsoft.Research.AbstractIL.Diagnostics
F#*)

open Ildiag
open List
open Range
open Ast
open Tast
open Tastops
open Env
open Lib
open Layout
open Il
open Typrelns
open Infos


(*--------------------------------------------------------------------------
!* NOTES: byref safety checks
 *--------------------------------------------------------------------------*)

(*
  The .NET runtime has safety requirements on the use of byrefs.
  These include:
    A1: No generic type/method can be instantiated with byref types (meaning contains byref type).
    A2: No object field may be byref typed.

  In F# TAST level, byref types can be introduced/consumed at:
    B1: lambda ... (v:byref<a>) ...         -- binding sites for values.
    B2: &m                                  -- address of operator, where m is local mutable or reference cell.
    B3: ms.M()                              -- method calls on mutable structs.
    B4: *br                                 -- dereference byref
    B5: br <- x                             -- assign byref
    B6: expr@[byrefType]                    -- any type instantiation could introduce byref types.
    B7: asm                                 -- TExpr_asm forms that create/consume byrefs.
        a) I_ldfld <byref> expr
        b) I_stfld <byref>
        c) others TBD... work in progress.

  Closures imply objects.
  Closures are either:
    a) explicit lambda expressions.
    b) functions partially applied below their known arity.
 
  Checks:
    C1: check no instantiation can contain byref types.
    C2: check type declarations to ensure no object field will have byref type.
    C3: check no explicit lambda expressions capture any free byref typed expression.    
    C4: check byref type expr occur only as:
        C4.a) arg to functions occuring within their known arity.
        C4.b) arg to IL method calls, e.g. arising from calls to instance methods on mutable structs.
        C4.c) arg to property getter on mutable struct (record field projection)
        C4.d) rhs of byref typed binding (aliasing).
              Note [1] aliasing should not effect safety. The restrictions on RHS byref will also apply to alias.
              Note [2] aliasing happens in the generated hash/compare code.

  Check commentary:
    The C4 checks ensure byref expressions are only passed directly as method arguments (or aliased).
    The C3 check ensures byref expressions are never captured, e.g. passed as direct method arg under a capturing thunk.
    The C2 checks no type can store byrefs (C4 ensures F# code would never actually store them).
    The C1 checks no generic type could be instanced to store byrefs.
*)
let checkingByref = true


(*--------------------------------------------------------------------------
!* check environment
 *--------------------------------------------------------------------------*)

type env = { rights : Infos.accessorDomain  } 
type cenv = { g: tcGlobals; amap: Import.importMap; denv: displayEnv; viewCcu : ccu }
let mk_cenv  g amap viewCcu denv =  { g =g ; amap=amap; denv=denv; viewCcu= viewCcu}

let modul_rights cpath = Infos.AccessibleFrom (cpath,None)
let tycon_rights tcref = AccessibleFrom (cpath_of_tycon (deref_tycon tcref),Some(tcref))



(*--------------------------------------------------------------------------
!* check for byref types
 *--------------------------------------------------------------------------*)

let contains_byref_ty cenv typ = 
    let res = ref false in 
    let visitType ty = if is_byref_ty cenv.g ty then res := true in
    let visitTypar tp = () in
    typ |> iterType (visitType,visitTypar);
    !res

(*--------------------------------------------------------------------------
!* check captures under lambdas
 *--------------------------------------------------------------------------*)
  
let check_escapes cenv allowProtected syntacticArgs body =
  (* This is the definition of what can/can't be free in a lambda expression. This is checked at lambdas OR TBind(v,e) nodes OR TMethod nodes. *)
  (* For TBind(v,e) nodes we may know an 'arity' which gives as a larger set of legitimate syntactic arguments for a lambda. *)
  (* For TMethod(v,e) nodes we always know the legitimate syntactic arguments. *)
  let cant_be_free v = 
     (* First, if v is a syntactic argument, then it can be free since it was passed in. *)
     (* The following can not be free: *)
     (*   a) "Local" mutables, being mutables such that: *)
     (*         i)  the mutable has no arity (since arity implies top-level storage, top level mutables...) *)
     (*             Note: "this" arguments to instance members on mutable structs are mutable arguments. *)
     (*   b) BaseVal can never escape. *)
     (*   c) Byref typed values can never escape. *)
     
     (* These checks must correspond to the tests governing the error messages below. *)
     let passedIn = gen_mem local_vref_eq v syntacticArgs in
     if passedIn then
       false
     else
       (mutability_of_val v = Mutable && isNone (arity_of_val v)) ||
       (base_of_val v = BaseVal  && not passedIn) ||
       (is_byref_ty cenv.g (type_of_val v))
  in
  let frees = (free_in_expr body) in                                                   
  let fvs   = frees.free_locvals in
  if not allowProtected && frees.uses_method_local_constructs then
    errorR(Error("A protected member is called or a base variable is being used. This is currently only allowed in the direct implementation of members since they could escape their object scope",range_of_expr body))
  else if Zset.exists cant_be_free fvs then 
    let v =  find cant_be_free (Zset.elements fvs) in
    (* byref error before mutable error (byrefs are mutable...). *)
    if (is_byref_ty cenv.g (type_of_val v)) then
        errorR(Error("The byref-typed variable '"^(display_name_of_val v)^"' is used in an invalid way. Byrefs may not be captured by closures or passed to inner functions",range_of_expr body))
        (* Inner functions are not guaranteed to compile to method with a predictable arity (number of arguments). *)
        (* As such, partial applications involving byref arguments could lead to closures containnig byrefs. *)
        (* For safety, such functions are assumed to have no known arity, and so can not accept byrefs. *)
    else if mutability_of_val v = Mutable then 
      errorR(Error("The mutable variable '"^(display_name_of_val v)^"' is used in an invalid way. Mutable variables may not be captured by closures. Consider eliminating this use of mutation or using a heap-allocated mutable reference cell via 'ref' and '!'",range_of_expr body))
    else if base_of_val v = BaseVal then
        errorR(Error("The base variable '"^(display_name_of_val v)^"' is used in an invalid way. Base variables may not be captured by closures",range_of_expr body))
    else
        errorR(InternalError("The variable '"^(display_name_of_val v)^"' is used in an invalid way",range_of_expr body)) (* <- should be dead code, unless governing tests change *)


(*--------------------------------------------------------------------------
!* check type instantiations
 *--------------------------------------------------------------------------*)


let check_ty_for_access (cenv:cenv) valName valAcc m ty =
   let visitType ty = 
       if is_stripped_tyapp_typ ty then (
           let tcref = tcref_of_stripped_typ ty in 
           let tyconAcc = access_of_tycon (deref_tycon tcref) in 
           if isLessAccessible tyconAcc valAcc then 
               errorR(Error(Printf.sprintf "The type '%s' is less accessible than the value, member or type '%s' it is used in" (display_name_of_tcref tcref) valName,m)) 
       ) in
   let visitTypar tp = () in
   ty |> iterType (visitType,visitTypar)

let check_ty_no_byrefs (*route*) (cenv:cenv) m ty =
  if checkingByref then (
    if contains_byref_ty cenv ty then (
      errorR(Error("A type instantiation involves a byref type. This is not permitted by the .NET runtime",m))
    );
  )


let check_ty_byrefs_ok (*route*) (cenv:cenv) ty =
  ()

let check_tinst_no_byrefs (*route*) (cenv:cenv) ad m tyargs =
  tyargs |> List.iter (check_ty_no_byrefs cenv m)

let check_tinst_byrefs_ok (*route*) (cenv:cenv) ad m tyargs =
  tyargs |> List.iter (check_ty_byrefs_ok cenv)


(*--------------------------------------------------------------------------
!* check exprs etc
 *--------------------------------------------------------------------------*)
  
type context = 
    | KnownArityTuple of int    (* Tuple of contexts allowing byref typed expr *)
    | DirectArg                 (* Context allows for byref typed expr *)
    | GeneralContext            (* General (byref type expr not allowed) *)
let mkKnownArity n = if n=1 then DirectArg else KnownArityTuple n

let argAritiesOfVal vref = match arity_of_vref vref with
                           | Some arity_info -> List.map mkKnownArity (TopValData.aritiesOfArgs arity_info)
                           | None -> []  
let rec argAritiesOfFunExpr x =
    match x with 
    | TExpr_val (vref,_,_)         -> argAritiesOfVal     vref      (* recognise val *)
    | TExpr_link eref              -> argAritiesOfFunExpr !eref     (* step through reclink  *)
    | TExpr_app(f,fty,tyargs,[],m) -> argAritiesOfFunExpr f         (* step through instantiations *)
    | _                            -> []

let rec check_expr   (cenv:cenv) (env:env) expr = check_expr_context cenv env expr GeneralContext
and check_expr_context (cenv:cenv) (env:env) expr (context:context) =    
    (* dprintf1 "check_expr: %s\n" (showL(exprL expr)); *)
    let expr = strip_expr expr in
    match expr with
    | TExpr_seq (e1,e2,flag,m) -> 
        check_expr cenv env e1; 
        check_expr cenv env e2
    | TExpr_let (bind,body,m,_) ->  
        check_bind cenv env bind ; 
        check_expr cenv env body
    | TExpr_const (c,m,ty) -> 
        check_ty_byrefs_ok (*(fun() -> sprintf "expr:const")*) cenv ty 
    
    | TExpr_val (v,vFlags,m) -> 
          if cenv.g.vref_eq v cenv.g.addrof_vref then errorR(Error("First-class uses of the address-of operators are not permitted",m));
          if cenv.g.vref_eq v cenv.g.addrof2_vref then errorR(Error("First-class uses of the address-of operators are not permitted",m));
          if checkingByref && is_byref_ty cenv.g (type_of_vref v) then (
            (* byref typed val can only occur in permitting contexts *)
            if context <> DirectArg then errorR(Error("The byref typed value '" ^ display_name_of_vref v ^ "' may not be used at this point",m))
          )
    | TExpr_hole (m,ty) -> 
          check_ty_no_byrefs (*(fun() -> sprintf "expr:quote")*) cenv m ty 
    | TExpr_quote(raw,ast,m,ty) -> 
          check_expr cenv env ast;
          begin 
              if not (Zset.is_empty (free_in_expr ast).free_tyvars.free_loctypars) then 
                  error(Error("This quotation term is generic, i.e. contains references to generic parameters from the context. Consider constraining the types of parameters so that this quotation is not generic",m))
              else 
                 (try Creflect.convExpr (Creflect.mk_cenv (cenv.g, cenv.amap, cenv.viewCcu, nng)) Creflect.empty_env ast  |> ignore
                  with Creflect.InvalidQuotedTerm e -> error(e)) 
          end;

          check_ty_no_byrefs (*(fun() -> sprintf "expr:quote")*) cenv m ty;
    | TExpr_obj (_,typ,basev,basecall,overrides,iimpls,m,_) -> 
          check_expr cenv env basecall;
          check_methods cenv env basev overrides ;
          check_iimpls cenv env basev iimpls;
    | TExpr_op (c,tyargs,args,m) ->
          check_op cenv env (c,tyargs,args,m) context
    | TExpr_app(f,fty,tyargs,argsl,m) ->
          if checkingByref then check_tinst_no_byrefs (*(fun() -> "app")*) cenv env.rights m tyargs;
          check_ty_byrefs_ok (*(fun() -> sprintf "expr:quote")*) cenv fty;
          check_tinst_byrefs_ok (*(fun() -> sprintf "expr:quote")*) cenv env.rights m tyargs;
          check_expr cenv env f;
          check_exprs_contexts cenv env argsl (argAritiesOfFunExpr f)
    (* REVIEW: fold the next two cases together *)
    | TExpr_lambda(lambda_id,basevopt,argvs,body,m,rty,_) -> 
        let arity_info = TopValInfo (0,[argvs |> map (fun _ -> TopValData.unnamedTopArg1)],TopValData.unnamedRetVal) in 
        check_ty_byrefs_ok (*(fun() -> sprintf "expr:quote")*) cenv rty;
        let ty = mk_multi_lambda_ty argvs rty in 
        check_lambdas None cenv env false arity_info expr ty
    | TExpr_tlambda(lambda_id,tps,body,m,rty,_)  -> 
        let arity_info = TopValInfo (length tps,[],TopValData.unnamedRetVal) in
        check_ty_byrefs_ok (*(fun() -> sprintf "expr:quote")*) cenv rty;
        let ty = try_mk_forall_ty tps rty in 
        check_lambdas None cenv env false arity_info expr ty
    | TExpr_tchoose(tps,e1,m)  -> 
        check_expr cenv env e1 
    | TExpr_match(exprm,dtree,targets,m,ty,_) -> 
        check_ty_no_byrefs (*(fun() -> sprintf "expr:quote")*) cenv m ty;
        check_dtree cenv env dtree;
        check_targets cenv env m ty targets;
    | TExpr_letrec (binds,e,m,_) ->  
        check_binds cenv env binds;
        check_expr cenv env e
    | TExpr_static_optimization (constraints,e2,e3,m) -> 
        check_expr cenv env e2;
        check_expr cenv env e3;
        constraints |> iter (fun (TTyconEqualsTycon(ty1,ty2)) -> 
            check_ty_no_byrefs cenv m ty1;
            check_ty_no_byrefs cenv m ty2)
    | TExpr_link eref -> failwith "unexpected reclink"

and check_methods cenv env basevopt l = List.iter (check_method cenv env basevopt) l
and check_method cenv env basevopt (TMethod(slotsig,tps,vs,e,m) as tmethod) = 
    check_escapes cenv true (match basevopt with Some x -> x::vs | None -> vs) e;
    check_expr cenv env e

and check_iimpls cenv env basevopt l = List.iter (check_iimpl cenv env basevopt) l
and check_iimpl cenv env basevopt (ty,overrides) = 
    let env = { env with rights = tycon_rights (tcref_of_stripped_typ ty) } in
    check_methods cenv env basevopt overrides 

and check_op cenv env (op,tyargs,args,m) context =
    (* Special cases *)
    match op,tyargs,args,context with 
    (* Handle these as special cases since mutables are allowed inside their bodies *)
    | TOp_while,_,[TExpr_lambda(_,_,[_],e1,_,_,_);TExpr_lambda(_,_,[_],e2,_,_,_)],_  
    | TOp_try_finally,[_],[TExpr_lambda(_,_,[_],e1,_,_,_); TExpr_lambda(_,_,[_],e2,_,_,_)],_ ->
        check_tinst_no_byrefs (*(fun() -> "op:try_finally")*) cenv env.rights m tyargs; 
        check_exprs cenv env [e1;e2]
    | TOp_for(_),_,[TExpr_lambda(_,_,[_],e1,_,_,_);TExpr_lambda(_,_,[_],e2,_,_,_);TExpr_lambda(_,_,[_],e3,_,_,_)],_  
    | TOp_try_catch,[_],[TExpr_lambda(_,_,[_],e1,_,_,_); TExpr_lambda(_,_,[_],e2,_,_,_); TExpr_lambda(_,_,[_],e3,_,_,_)],_ ->
        check_tinst_no_byrefs (*(fun() -> "op:try_catch")*) cenv env.rights m tyargs;
        check_exprs cenv env [e1;e2;e3]
    | TOp_ilcall ((virt,protect,valu,newobj,superInit,prop,isDllImport,boxthis,mref),enclTypeArgs,methTypeArgs,tys),_,_,_ ->
        check_tinst_no_byrefs (*(fun() -> "ilcall:tyargs")*)       cenv env.rights m tyargs;
        check_tinst_no_byrefs (*(fun() -> "ilcall:enclTypeArgs")*) cenv env.rights m enclTypeArgs;
        check_tinst_no_byrefs (*(fun() -> "ilcall:methTypeArgs")*) cenv env.rights m methTypeArgs;
        check_tinst_no_byrefs (*(fun() -> "ilcall:tys")*)          cenv env.rights m tys;
        check_exprs_directArgs cenv env args  
    | TOp_tuple,_,_,KnownArityTuple nArity ->           (* tuple expression in known tuple context             *)
        if List.length args <> nArity then errorR(InternalError("Tuple arity does not correspond to planned function argument arity",m));
        (* This tuple should not be generated. The known function arity means it just bundles arguments. *)
        check_exprs_directArgs cenv env args  
    | TOp_lval_op(LGetAddr,v),_,_,arity -> 
        if checkingByref then
          if arity = DirectArg then       
            check_exprs cenv env args                   (* Address-of operator generates byref, and context permits this. *)
          else            
            errorR(Error("The address of the variable '" ^ display_name_of_vref v ^"' may not be used at this point",m))            
    | TOp_field_get rf,_,[arg1],arity -> 
        check_tinst_no_byrefs (*(fun() -> sprintf "op:field_get")*) cenv env.rights m tyargs;
        check_exprs_directArgs cenv env [arg1]          (* See mk_recd_field_get_via_expra -- byref arg1 when #args =1 *)
                                                        (* Property getters on mutable structs come through here. *)
    | TOp_field_set rf,_,[arg1;arg2],arity -> 
        check_tinst_no_byrefs (*(fun() -> sprintf "op:field_set")*) cenv env.rights m tyargs;
        check_exprs_directArgs cenv env [arg1];         (* See mk_recd_field_set_via_expra -- byref arg1 when #args=2 *)
        check_exprs            cenv env [arg2]          (* Property setters on mutable structs come through here (TBC). *)
    | TOp_coerce,[ty1;ty2],[x],arity ->
        check_tinst_no_byrefs (*(fun() -> sprintf "op:field_set")*) cenv env.rights m tyargs;
        check_expr_context cenv env x context
    | TOp_field_get_addr rfref,tyargs,[],_ ->
        if checkingByref && context <> DirectArg then
          errorR(Error("The address of the static field '"^name_of_rfref rfref^"' may not be used at this point",m));
        check_tinst_no_byrefs (*(fun() -> sprintf "op:field_get_addr")*) cenv env.rights m tyargs
        (* NOTE: there are no arg exprs to check in this case *)
    | TOp_field_get_addr rfref,tyargs,[rx],_ ->
        if checkingByref && context <> DirectArg then
          errorR(Error("The address of the field '"^name_of_rfref rfref^"' may not be used at this point",m));
        (* This construct is used for &(rx.rfield) and &(rx->rfield). Relax to permit byref types for rx. [See Bug 1263]. *)
        check_tinst_no_byrefs (*(fun() -> sprintf "op:field_get_addr")*) cenv env.rights m tyargs;
        check_expr_context cenv env rx DirectArg (* allow rx to be byref here *)
    | TOp_asm (instrs,tys),_,_,_  ->
        check_tinst_byrefs_ok (*(fun() -> sprintf "op:other %A %s" op (String.concat "," (List.map (exprL >> showL) args)))*) cenv env.rights m tys;
        check_tinst_no_byrefs (*(fun() -> sprintf "op:other %A %s" op (String.concat "," (List.map (exprL >> showL) args)))*) cenv env.rights m tyargs;
        begin
            match instrs,args with
            | [ I_stfld (alignment,vol,fspec) ],[lhs;rhs] ->
                check_expr_context cenv env lhs DirectArg; (* permit byref for lhs lvalue *)
                check_expr         cenv env rhs                
            | [ I_ldfld (alignment,vol,fspec) ],[lhs] ->
                check_expr_context cenv env lhs DirectArg  (* permit byref for lhs lvalue *)
            | [ I_ldflda (fspec) ],[lhs] ->
                if checkingByref && context <> DirectArg then
                  errorR(Error("The address of the field '"^name_of_fspec fspec^"' may not be used at this point",m));
                check_expr_context cenv env lhs DirectArg  (* permit byref for lhs lvalue *)
            | [ I_ldsflda (fspec) ],[] ->
                if checkingByref && context <> DirectArg then
                  errorR(Error("The address of the field '"^name_of_fspec fspec^"' may not be used at this point",m));
            | instrs ->
                check_exprs cenv env args  
        end
    | (   TOp_tuple
        | TOp_uconstr _
        | TOp_exnconstr _
        | TOp_array
        | TOp_bytes _
        | TOp_recd _
        | TOp_field_set _
        | TOp_constr_tag_get _
        | TOp_constr_field_get _
        | TOp_constr_field_set _
        | TOp_exnconstr_field_get _
        | TOp_exnconstr_field_set _
        | TOp_tuple_field_get _
        | TOp_get_ref_lval 
        | TOp_trait_call _
        | _ (* catch all! *)
        ),_,_,_ ->    
        check_tinst_no_byrefs (*(fun() -> sprintf "op:other %A %s" op (String.concat "," (List.map (exprL >> showL) args)))*) cenv env.rights m tyargs;
        check_exprs cenv env args 

and check_lambdas memInfo cenv env inlined arity_info e ety =
  (* The arity_info here says we are _guaranteeing_ to compile a function value *)
  (* as a .NET method with precisely the corresponding argument counts. *)
  match e with
  | TExpr_tchoose(tps,e1,m)  -> 
      check_lambdas memInfo cenv env inlined arity_info e1 ety      

  | TExpr_lambda (lambda_id,_,_,_,m,_,_)  
  | TExpr_tlambda(lambda_id,_,_,m,_,_) ->

      let tps,basevopt,vsl,body,bodyty = dest_top_lambda_upto cenv.g cenv.amap arity_info (e, ety) in
      let vspecs = (Option.to_list basevopt @ List.concat vsl) in 
      vspecs |> iter (check_val cenv env);
      
      (* Allow access to protected things within members *)
      let env = 
          match memInfo with 
          | None -> env 
          | Some membInfo -> { env with rights=tycon_rights membInfo.vspr_apparent_parent } in
      
      check_escapes cenv (isSome(memInfo)) vspecs body;
      check_expr cenv env body;
      if checkingByref && not inlined && contains_byref_ty cenv bodyty then       
        errorR(Error("A method return type would contain byrefs which is not permitted",m))
  | _ -> 
    if checkingByref && not inlined && is_byref_ty cenv.g ety then
      check_expr_context cenv env e DirectArg           (* allow byref to occur as RHS of byref binding. *)
    else 
      check_expr cenv env e

and check_exprs_contexts cenv env exprs arities =
    let arities = Array.of_list arities in
    let argArity i = if i < Array.length arities then arities.(i) else GeneralContext in
    list_iteri (fun i exp -> check_expr_context cenv env exp (argArity i)) exprs

and check_exprs            cenv env exprs = iter (check_expr cenv env) exprs
and check_exprs_directArgs cenv env exprs = iter (fun x -> check_expr_context cenv env x DirectArg) exprs
and check_targets cenv env m ty targets = Array.iter (check_target cenv env m ty) targets

and check_target cenv env m ty (TTarget(vs,e)) = check_expr cenv env e;

and check_dtree cenv env x =
  match x with 
  | TDSuccess (es,n) -> check_exprs cenv env es;
  | TDBind(bind,rest) -> check_bind cenv env bind; check_dtree cenv env rest 
  | TDSwitch (e,cases,dflt,m) -> check_switch cenv env (e,cases,dflt,m)

and check_switch cenv env (e,cases,dflt,m) =
  check_expr cenv env e;
  iter (fun (TCase(discrim,e)) -> check_dtree cenv env e) cases;
  Option.iter (check_dtree cenv env) dflt

and check_attrib cenv env (Attrib(k,args,props)) = 
  check_exprs cenv env args;
  props |> iter (fun (nm,ty,flg,expr) -> check_expr cenv env expr)
  
and check_attribs cenv env attribs = iter (check_attrib cenv env) attribs

and check_topValInfo cenv env (TopValInfo(_,args,ret)) =
    args |> List.iter (List.iter (check_topArgInfo cenv env));
    ret |> check_topArgInfo cenv env;

and check_topArgInfo cenv env (TopArgData(attribs,_)) = 
    check_attribs cenv env attribs

and check_val cenv env v =
    v |> attribs_of_val |> check_attribs cenv env;
    v |> arity_of_val |> Option.iter (check_topValInfo cenv env);
    v |> type_of_val |> check_ty_byrefs_ok cenv 

and check_bind cenv env (TBind(v,e) as bind) =
    v |> attribs_of_val |> check_attribs cenv env;
    v |> arity_of_val |> Option.iter (check_topValInfo cenv env);
    if modbind_of_val v or isSome(member_info_of_val v) then 
        v |> type_of_val |> check_ty_for_access cenv (name_of_val v) (access_of_val v) (range_of_val v);
    
    if isSome (pubpath_of_val v) then (
        if fsthing_has_attrib cenv.g cenv.g.attrib_ReflectedDefinitionAttribute (attribs_of_val v) then
           (* If we've already recorded a definition then skip this *)
            match published_closed_defn_of_val v with 
            | None -> (data_of_val v).val_defn <- Some e
            | Some _ -> ()

    );
    let arity_info  = match chosen_arity_of_bind bind with Some info -> info | _ -> TopValData.emptyTopValData in
    let inlined     = inlineFlag_of_val v = AlwaysInline || inlineFlag_of_val v = PseudoValue in
      (* certain inline functions are permitted to have byref return types, since they never compile to records. *)
      (* e.g. for the byref operator itself, &. *)
    check_lambdas (member_info_of_val v) cenv env inlined arity_info e (type_of_val v);

and check_binds cenv env xs = iter (check_bind cenv env) xs

(*--------------------------------------------------------------------------
!* check tycons
 *--------------------------------------------------------------------------*)
  
let check_tycon_rfield cenv env tycon (rfield:recdfield_spec) = 
    check_ty_for_access cenv (name_of_rfield rfield) (access_of_rfield rfield) (range_of_rfield rfield) rfield.rfield_type ;
    check_attribs cenv env (pattribs_of_rfield rfield);
    check_attribs cenv env (fattribs_of_rfield rfield);
    if contains_byref_ty cenv rfield.rfield_type then
      errorR(Error("A type would store a byref typed value. This is not permitted by the .NET runtime",range_of_tycon tycon))

let check_tycon cenv env tycon =
    check_attribs cenv env (attribs_of_tycon tycon);
    (* Considers TFsObjModelRepr, TRecdRepr and TFiniteUnionRepr. *)
    (* [Review] are all cases covered: TIlObjModelRepr,TAsmRepr *)
    tycon |> rfields_array_of_tycon |> Array.iter (check_tycon_rfield cenv env tycon);
    if is_union_tycon tycon then (                            (* This covers finite unions. *)
      uconstrs_of_tycon tycon |> List.iter (fun uc ->
          check_attribs cenv env (attribs_of_uconstr uc);
          uc |> rfields_of_uconstr |> List.iter (check_tycon_rfield cenv env tycon))
    );
    match abbrev_of_tycon tycon with                          (* And type abbreviations *)
     | None     -> ()
     | Some typ -> 
         if contains_byref_ty cenv typ then
           errorR(Error("The type abbreviation contains byrefs. This is not permitted by F#",range_of_tycon tycon))

let check_tycons cenv env tycons = List.iter (check_tycon cenv env) tycons


(*--------------------------------------------------------------------------
!* check modules
 *--------------------------------------------------------------------------*)

let rec check_mexpr cenv env x = 
    match x with  
    | TMTyped(mty,def,m) -> check_mdef cenv env def
    
and check_mdefs cenv env x = iter (check_mdef cenv env) x

and check_mdef cenv env x = 
    match x with 
    | TMDefRec(tycons,binds,m) -> check_tycons cenv env tycons; check_binds cenv env binds 
    | TMDefLet(bind,m)  -> check_bind cenv env bind 
    | TMAbstract(def)  -> check_mexpr cenv env def
    | TMDefModul(TMBind(tycon, rhs)) -> 
        check_tycon cenv env tycon;
        check_mdef cenv { env with rights=modul_rights (cpath_of_tycon tycon) } rhs 
    | TMDefs(defs) -> check_mdefs cenv env defs 

let check_top_impl cenv (TImplFile(_,mexpr)) =
    check_mexpr cenv { rights = AccessibleFromEverywhere } mexpr 


