// (c) Microsoft Corporation 2005-2007. 

#light

/// Tags used to mark collection types with information concerning the hash, equality and/or comparison functions used
namespace Microsoft.FSharp.Collections.Tags
    open Microsoft.FSharp.Core
    open Microsoft.FSharp.Collections
    open Microsoft.FSharp.Primitives.Basics
    open System

    #if CLI_AT_MOST_1_1
    open Microsoft.FSharp.Compatibility
    #else
    open System.Collections.Generic
    #endif

    /// This tag indicates the collection uses structural hashing on keys, except where the structural hashing
    /// semantics has been altered through re-implementations of the IStructuralHash interface
    type StructuralHash<'key>       = 
        new: unit ->  StructuralHash<'key>     
        interface IEqualityComparer<'key> 

    /// This tag indicates the collection uses physical hashing (hash on reference 
    /// identity of objects, and the contents of value types).  
    /// i.e. for value types use GetHashCode and Object.Equals (if no other optimization available),
    /// and for reference types use System.Runtime.CompilerServices.RuntimeHelpers.GetHashCode and 
    /// reference equality. 
    type ReferenceHash<'key>        = 
        new: unit ->  ReferenceHash<'key>      
        interface IEqualityComparer<'key> 

    /// This tag indicates the collection uses structural comparison on keys, except where the structural comparison
    /// semantics has been altered for particular types through re-implementations of the IComparer interface.
    type StructuralComparer<'key>   = 
        new: unit ->  StructuralComparer<'key> 
        interface IComparer<'key>         
    
    
    module Optimizations = 
        /// This tag represents an internal optimization and should not be used directly by the programmer
        type StructuralHashInt       = 
            inherit StructuralHash<int>       
            new: unit ->  StructuralHashInt    
            interface IEqualityComparer<int>  

        /// This tag represents an internal optimization and should not be used directly by the programmer
        type StructuralComparerInt  = 
            inherit StructuralComparer<int> 
            new: unit ->  StructuralComparerInt    
            interface IComparer<int>          

/// This namespace contains some common collections in an object-oriented 
/// style well suited for use from F#.  

namespace Microsoft.FSharp.Collections
    open Microsoft.FSharp.Core
    open Microsoft.FSharp.Collections
    open Microsoft.FSharp.Primitives.Basics
    open System

#if CLI_AT_MOST_1_1
    open Microsoft.FSharp.Compatibility
#else
    open System.Collections.Generic
#endif

    /// Common notions of comparison identity used with sorted data structures.
    module ComparisonIdentity = 
      
        /// Structural comparison.  Compare using LanguagePrimitives.GenericComparison.
        val inline Structural<'key> : Tags.StructuralComparer<'key> 
        
        /// Compare using the given comparer comparer function
        val inline FromFunction : ('key -> 'key -> int) -> IComparer<'key> 

        [<System.Obsolete("This function has been renamed to ComparisonIdentity.FromFunction")>]
        val inline Custom : ('key -> 'key -> int) -> IComparer<'key> 
        
        /// Compare using any implementation of System.IComparable interfaces. This differs from structural comparison
        /// because arrays are not compared structurally, and many optimizations are not applied.
        [<System.Obsolete("This function is no longer supported. Consider creating an explicit implementation of IComparer using an object expression")>]
        val inline ViaIComparable<'key> : IComparer<'key> 



    /// Common notions of value identity used with hash tables.
    module HashIdentity = 

        /// Generic hashing.  Hash using LanguagePrimitives.GenericEquality and LanguagePrimitives.GenericComparer.
        /// i.e. for value types use GetHashCode and Object.Equals (if no other optimization available),
        /// and for reference types use the F# cooperative structural term hashing scheme.  Object.GetHashCode(obj)
        /// is used if no other choice is available.
        val inline Structural<'key> : Tags.StructuralHash<'key> 
        
        /// Physical hashing (hash on reference identity of objects, and the contents of value types).  
        /// Hash using LanguagePrimitives.PhysicalEquality and LanguagePrimitives.PhysicalHash,
        /// i.e. for value types use GetHashCode and Object.Equals (if no other optimization available),
        /// and for reference types use System.Runtime.CompilerServices.RuntimeHelpers.GetHashCode and 
        /// reference equality. 
        val inline Reference<'key>   : Tags.ReferenceHash<'key> 
        
        /// Physical hashing (hash on reference identity of objects, and the contents of value types).  
        /// Hash using LanguagePrimitives.PhysicalEquality and LanguagePrimitives.PhysicalHash,
        /// i.e. for value types use GetHashCode and Object.Equals (if no other optimization available),
        /// and for reference types use System.Runtime.CompilerServices.RuntimeHelpers.GetHashCode and 
        /// reference equality. 

        [<System.Obsolete("Consider creating an explicit implementation of IEqualityComparer using an object expression instead")>]
        val inline FromFunctions<'key> : ('key -> int) -> ('key -> 'key -> bool) -> IEqualityComparer<'key> 

        [<Obsolete("This function has been renamed to 'FromFunctions'")>]
        val inline FromFunction<'key> : ('key -> int) -> ('key -> 'key -> bool) -> IEqualityComparer<'key> 

        [<System.Obsolete("This function has been renamed to HashIdentity.FromFunction")>]
        val inline Custom<'key> : ('key -> int) -> ('key -> 'key -> bool) -> IEqualityComparer<'key> 
        
        /// Hash using Object.GetHashCode and Object.Equals(Object)
        [<System.Obsolete("Consider creating an explicit implementation of IEqualityComparer using an object expression instead")>]
        val inline Object<'key> : IEqualityComparer<'key> 

        /// Hash using Object.GetHashCode and Object.Equals(Object)
        [<System.Obsolete("Consider creating an explicit implementation of IEqualityComparer using an object expression instead")>]
        val inline ViaGetHashCode<'key> : IEqualityComparer<'key> 


    /// Sinks recording statistical properties of one or more hash tables.
    type HashStats with
        /// Create a new HashStats sink.
        static member Create : unit -> HashStats
        /// The total number of resizes of all hash tables traced by this sink.
        member NumResizes: int
        /// The total number of tables traced by this sink.
        member NumTables: int
        /// The longest ever chain length sampled in all tables traced by this sink.
        /// Chain levels are typically sampled only on resizes.
        member LongestChainLength: int

    /// Hash tables, by default based on F# structural "hash" and (=) functions. 
    /// This is precisely the same type as that manipulated by 
    /// the functions in [[Microsoft.FSharp.Compatibility.OCaml.Hashtbl]].
    /// The table may map a single key to multiple bindings.
    type HashMultiMap<'key,'a> with 
        /// Create a new empty mutable hash table 
        /// with key hash/equality based on the F# structural "hash" and (=) functions.
        ///
        /// This function is always inlined to give compilers the opportunity to optimize the
        /// structural hash/equality functions based on any available compile-time knowledge 
        /// of the key type at the call site.
        static member inline Create  : unit -> HashMultiMap<'key,'a>

        /// Create a new empty mutable hash table with an internal bucket array of the given approximate size
        /// and with key hash/equality based on the F# structural "hash" and (=) functions
        ///
        /// This function is always inlined to give compilers the opportunity to optimize the
        /// structural hash/equality functions based on any available compile-time knowledge 
        /// of the key type at the call site.
        static member inline Create  : int -> HashMultiMap<'key,'a>
        
        /// Create a new empty mutable hash table with an internal bucket array of the given approximate size
        /// and with the given key hash/equality functions
        static member Create: IEqualityComparer<'key> * int             -> HashMultiMap<'key,'a>

        /// Create a new empty mutable hash table with an internal bucket array of the given approximate size
        /// and with the given key hash/equality functions and the given HashStats collector
        [<Obsolete("HashStats are deprecated. The HashStats argument will be ignored")>]
        static member Create: IEqualityComparer<'key> * HashStats * int -> HashMultiMap<'key,'a>

        /// Build a map that contains the bindings of the given IEnumerable
        [<OverloadID("Create_seq")>]
        static member Create: seq<'key * 'a> -> HashMultiMap<'key,'a>
        
        /// Make a shallow copy of the collection
        member Copy    : unit    -> HashMultiMap<'key,'a>
        /// Add a binding for the element to the table
        member Add     : 'key * 'a -> unit
        /// Clear all elements from the collection
        member Clear   : unit    -> unit
        /// Test if the collection contains any bindings for the given element
        member Contains: 'key      -> bool
        /// Remove the latest binding (if any) for the given element from the table
        member Remove  : 'key      -> unit
        /// Replace the latest binding (if any) for the given element.
        member Replace : 'key * 'a -> unit

        /// Lookup or set the given element in the table.  Raise [[Not_found]]/[[KeyNotFoundException]] if the element is not found.
        member Item : 'key -> 'a with get,set

        [<System.Obsolete("Consider using hashTable.[key] instead")>]
        member Find    : 'key      -> 'a
        /// Lookup the given element in the table, returning the result as an Option
        member TryFind : 'key      -> 'a option
        /// Find all bindings for the given element in the table, if any
        member FindAll : 'key      -> 'a list

        /// Apply the given function to each element in the collection threading the accumulating parameter
        /// through the sequence of function applications
        member Fold    : ('key -> 'a -> 'c -> 'c) -> 'c -> 'c

        /// The total number of elements in the hash table
        member Count   : int

        ///Apply the given function to each binding in the hash table 
        member Iterate : ('key -> 'a -> unit) -> unit

        /// Get the longest chain in the hash table as a list.  Use for debugging
        //  poorly performing hash functions
        [<Obsolete("GetLongestChain are deprecated. It will return the empty list")>]
        member GetLongestChain : unit -> ('key * 'a) list 

#if CLI_AT_MOST_1_1
#else
        interface IDictionary<'key, 'a>         
        interface ICollection<KeyValuePair<'key, 'a>> 
#endif
        interface IEnumerable<KeyValuePair<'key, 'a>>         
        interface System.Collections.IEnumerable 
        
    /// Mutable hash sets based by default on F# structural "hash" and (=) functions. Implemented via a hash table and/or Dictionary.
    type HashSet<'a>  with
        
        /// Create a new empty mutable hash set 
        /// with key hash/equality based on the F# structural "hash" and (=) functions
        ///
        /// This function is always inlined to give compilers the opportunity to optimize the
        /// structural hash/equality functions based on any available compile-time knowledge 
        /// of the key type at the call site.
        static member inline Create  : unit -> HashSet<'a>

        /// Create a new empty mutable hash set with an internal bucket array of the given approximate size
        /// and with key hash/equality based on the F# structural "hash" and (=) functions
        ///
        /// This function is always inlined to give compilers the opportunity to optimize the
        /// structural hash/equality functions based on any available compile-time knowledge 
        /// of the key type at the call site.
        static member inline Create  : int -> HashSet<'a>

        /// Create a new empty mutable hash set with an internal bucket array of the given approximate size
        /// and with the given key hash/equality functions 
        static member Create: IEqualityComparer<'a> * int             -> HashSet<'a>

        /// Create a new empty mutable hash set with an internal bucket array of the given approximate size
        /// and with the given key hash/equality functions and the given HashStats collector
        [<Obsolete("HashStats are deprecated. The HashStats argument will be ignored")>]
        static member Create: IEqualityComparer<'a> * HashStats * int -> HashSet<'a>
        
        /// Create a new mutable hash set containing elements drawn from the given sequence
        [<OverloadID("Create_seq")>]
        static member Create: seq<'a> -> HashSet<'a>
        
        /// Make a shallow copy of the set
        member Copy    : unit -> HashSet<'a>
        /// Add an element to the collection
        member Add     : 'a   -> unit
        /// Clear all elements from the set
        member Clear   : unit -> unit
        /// Test if the set contains the given element
        member Contains: 'a   -> bool
        /// Remove the given element from the set
        member Remove  : 'a   -> unit
        /// Apply the given function to the set threading the accumulating parameter
        /// through the sequence of function applications
        member Fold    : ('a -> 'b -> 'b) -> 'b -> 'b

        /// The total number of elements in the set
        member Count   : int

        /// Apply the given function to each binding in the hash table 
        member Iterate : ('a -> unit) -> unit

        /// Use to debug poorly performing hash tables
        [<Obsolete("GetLongestChain is deprecated. It will return the empty list")>]
        member GetLongestChain : unit -> 'a list 

#if CLI_AT_MOST_1_1
#else
        interface IEnumerable<'a> 
            
        interface System.Collections.IEnumerable 
#endif
  
    /// Immutable sets based on binary trees, where comparison is the
    /// F# structural comparison function, potentially using implementations
    /// of the IComparable interface on key values.
    ///
    /// See the Set module for further operations on sets.
    ///
    /// These sets can be used with elements of any type, but you should check that
    /// structural hashing and equality on the element type are correct for your type.  
    ///
    /// Efficiency: Structural comparison is relatively efficient but is not a suitable
    /// choice in all circumstances, e.g. it may not compare efficiently on 
    /// non-reference types and deeply-structured types. 
    type Set<'a> with

        /// A useful shortcut for Set.add.  Note this operation prodcues a new set
        /// and does not mutate the original set.  The new set will share many storage
        /// nodes with the original.  See the Set module for further operations on sets.
        member Add : 'a -> Set<'a>
        
        /// A useful shortcut for Set.remove.  Note this operation prodcues a new set
        /// and does not mutate the original set.  The new set will share many storage
        /// nodes with the original.  See the Set module for further operations on sets.
        member Remove : 'a -> Set<'a>
        
        /// A useful shortcut for Set.size.  See the Set module for further operations on sets.
        [<System.Obsolete("Please use the .Count property instead")>]
        member Size : int

        /// The number of elements in the set
        member Count : int
        
        /// A useful shortcut for Set.mem.  See the Set module for further operations on sets.
        member Contains : 'a -> bool
        
        /// A useful shortcut for Set.is_empty.  See the Set module for further operations on sets.
        member IsEmpty  : bool

        /// Apply the given function to each binding in the collection
        member Iterate : ('a -> unit) -> unit

        /// Apply the given accumulating function to all the elements of the set
        member Fold    : ('a -> 'b -> 'b) -> 'b -> 'b

        /// Build two new sets, one containing the elements for which the given predicate returns 'true',
        /// and the other the remaining elements.
        member Partition: ('a -> bool) -> Set<'a> * Set<'a>

        /// Return a new collection containing only the elements of the collection
        /// for which the given predicate returns "true"
        member Filter: ('a -> bool) -> Set<'a> 

        /// Return a new collection containing the results of applying the
        /// given function to each element of the input set
        member Map: ('a -> 'b) -> Set<'b> 

        /// Test if any element of the collection satisfies the given predicate.
        /// If the elements are "i0...iN" then computes "p i0 or ... or p iN".
        member Exists: ('a -> bool) -> bool

        /// Test if all elements of the collection satisfy the given predicate.
        /// If the elements are "i0...iN" and "j0...jN" then computes 
        /// "p i0 &amp;&amp; ... &amp;&amp; p iN".
        member ForAll: ('a -> bool) -> bool

        [<Obsolete("Consider using Set.Empty instead")>]
        static member Create: unit -> Set<'a> 

        /// Build a set that contains the same elements as the given IEnumerable
        static member Create: seq<'a> -> Set<'a> 

        /// Build a set that contains the same elements as the given array
        static member FromArray: 'a array -> Set<'a> 

        /// The empty set based on the default structural comparison operator
        static member Empty: unit -> Set<'a> 
        /// A singleton set based on the default structural comparison operator
        static member Singleton: 'a -> Set<'a> 
        /// Compares two sets and returns true if they are equal or false otherwise
        static member Equality : Set<'a> * Set<'a> -> bool
        /// Compares a and b and returns 1 if a &gt; b, -1 if b &lt; a and 0 if a = b
        static member Compare : a:Set<'a> * b:Set<'a> -> int
 
        /// Return a new set with the elements of the second set removed from the first.
        static member (-) : Set<'a> * Set<'a> -> Set<'a> 

        /// Compute the union of the two sets.
        static member (+) : Set<'a> * Set<'a> -> Set<'a> 

        /// Compute the intersection of the two sets.
        static member Intersection : Set<'a> * Set<'a> -> Set<'a> 

        /// Compute the intersection of N sets. At least one set must be given.
        static member Intersection : seq<Set<'a>> -> Set<'a> 

        /// Compute the union of the two sets.
        static member Union : Set<'a> * Set<'a> -> Set<'a>

        /// Compute the union of multiple sets.
        static member Union : seq<Set<'a>> -> Set<'a>

        /// Return a new set with the elements of the second set removed from the first.
        static member Difference: Set<'a> * Set<'a> -> Set<'a> 

        /// Returns the lowest element in the set according to the ordering being used for the set
        member Choose : 'a 

        /// Returns the lowest element in the set according to the ordering being used for the set
        member MinimumElement: 'a

        /// Returns the highest element in the set according to the ordering being used for the set
        member MaximumElement: 'a

        /// Returns the least element in the set that is greater than the given key 
        /// according to the ordering being used for the set
        member GetNextElement: 'a -> 'a option

        /// Returns the greatest element in the set that is less than the given key 
        /// according to the ordering being used for the set
        member GetPreviousElement: 'a -> 'a option

        /// Evaluates to "true" if all elements of the second set are in the first
        member IsSubsetOf: Set<'a> -> bool

        /// Evaluates to "true" if all elements of the first set are in the second
        member IsSupersetOf: Set<'a> -> bool

        /// The elements of the set as a list.
        member ToList : unit -> 'a list
        
        /// The elements of the set as an array.
        member ToArray: unit -> 'a array 

#if CLI_AT_MOST_1_1
#else
        interface ICollection<'a> 
#endif
        interface IEnumerable<'a> 
        interface System.Collections.IEnumerable 

        interface System.IComparable
        override Equals : obj -> bool
        
        //interface IStructuralHash

        [<Obsolete("This property will be removed in a later release")>]
        member CheckBalanceInvariant : bool

    /// Immutable maps.  Keys are ordered by F# structural comparison, potentially using implementations
    /// of the IComparable interface on key values.
    ///
    /// <performance> 
    ///   Maps based on structural comparison are  
    ///   efficient for small keys. They are not a suitable choice if keys are recursive data structures 
    ///   or require non-structural comparison semantics.
    /// </performance>
    type Map<'key,'a> with 
        /// Return a new map with the binding added to the given map.
        member Add: 'key * 'a -> Map<'key,'a>

        /// Return true if there are no bindings in the map.
        member IsEmpty: bool

        /// The empty map
        static member Empty: unit -> Map<'key,'a>

        /// Builds a map that contains all the elements of the list.
        static member FromList : ('key * 'a) list -> Map<'key,'a>

        [<Obsolete("Consider using Map.Empty instead")>]
        static member Create: unit -> Map<'key,'a>
        /// Build a map that contains the bindings of the given IEnumerable
        static member Create: seq<'key * 'a> -> Map<'key,'a>

        /// Test is an element is in the domain of the map
        member ContainsKey: 'key -> bool

        /// The number of bindings in the map
        member Count: int

        [<System.Obsolete("Consider using map.[key] instead")>]
        member Find: 'key -> 'a

        /// Lookup an element in the map, raising [[Not_found]]/[[KeyNotFoundException]] if no binding
        /// exists in the map.
        member Item : 'key -> 'a with get

        /// Search the map looking for the first element where the given function returns a [[Some]] value
        member First: ('key -> 'a -> 'b option) -> 'b option

        /// Return true if the given predicate returns true for all of the
        /// bindings in the map. Always returns true if the map is empty.
        member ForAll: ('key -> 'a -> bool) -> bool

        /// Return true if the given predicate returns true for one of the
        /// bindings in the map. Always returns false if the map is empty.
        member Exists: ('key -> 'a -> bool) -> bool

        /// Build a new map containing the bindings for which the given predicate returns 'true'.
        member Filter: ('key -> 'a -> bool) -> Map<'key,'a> 

        /// Fold over the bindings in the map.  
        member Fold: ('key -> 'a -> 'state -> 'state) -> 'state -> 'state

        /// Given the start and end points of a key range,
        /// Fold over the bindings in the map that are in the range,
        /// and the end points are included if present (the range is considered a closed interval).
        member FoldSection: 'key -> 'key -> ('key -> 'a -> 'state -> 'state) -> 'state -> 'state

        /// Fold over the bindings in the map.  
        member FoldAndMap: ('key -> 'a -> 'state -> 'b * 'state) -> 'state -> Map<'key,'b> * 'state

        /// Apply the given function to each binding in the dictionary
        member Iterate: ('key -> 'a -> unit) -> unit

        /// Build a new collection whose elements are the results of applying the given function
        /// to each of the elements of the collection. The index passed to the
        /// function indicates the index of element being transformed.
        member Map: ('key -> 'a -> 'b) -> Map<'key,'b>

        /// Build a new collection whose elements are the results of applying the given function
        /// to each of the elements of the collection.
        member MapRange: ('a -> 'b) -> Map<'key,'b>

        /// Build two new maps, one containing the bindings for which the given predicate returns 'true',
        /// and the other the remaining bindings.
        member Partition: ('key -> 'a -> bool) -> Map<'key,'a> * Map<'key,'a>

        /// Remove an element from the domain of the map.  No exception is raised if the element is not present.
        member Remove: 'key -> Map<'key,'a>

        /// Lookup an element in the map, returning a [[Some]] value if the element is in the domain 
        /// of the map and [[None]] if not.
        member TryFind: 'key -> 'a option

        /// The elements of the set as a list.
        member ToList : unit -> ('key * 'a) list
    
        /// The elements of the set as an array
        member ToArray: unit -> ('key * 'a) array 

#if CLI_AT_MOST_1_1
#else
        interface IDictionary<'key, 'a>         
        interface ICollection<KeyValuePair<'key, 'a>> 
#endif
        interface IEnumerable<KeyValuePair<'key, 'a>>         
        interface System.IComparable
        interface System.Collections.IEnumerable 
        override Equals : obj -> bool

    
    
    [<System.Obsolete("The type HashTable has been renamed to HashMultiMap because the type doesn't conform to the expected behaviour of .NET Dictionaries (it can bind keys to multiple values). Please adjust your references to this type")>]
    type HashTable<'a,'b> = HashMultiMap<'a,'b>
    
    [<System.Obsolete("Consider using Microsoft.FSharp.Collections.HashMultiMap or Microsoft.FSharp.Collections.Tagged.HashMultiMap")>]
    type CHashTable<'a,'b> = HashTable<'a,'b>
    [<System.Obsolete("Consider using Microsoft.FSharp.Collections.HashSet or Microsoft.FSharp.Collections.Tagged.HashSet")>]
    type CHashSet<'a> = HashSet<'a>

    [<System.Obsolete("Collections.IHashOps was an abbreviation for System.Collections.Generic.IEqualityComparer and is now deprecated. Please use that name instead")>]
    type IHashOps<'key> = IEqualityComparer<'key> 



namespace Microsoft.FSharp.Collections.Tagged
    open Microsoft.FSharp.Core
    open Microsoft.FSharp.Collections
    open Microsoft.FSharp.Primitives.Basics
    open System

    #if CLI_AT_MOST_1_1
    open Microsoft.FSharp.Compatibility
    #else
    open System.Collections.Generic
    #endif

    /// HashMultiMap, but where a constraint tag tracks information about the hash/equality functions used
    /// for the hashing. When the tag is Tags.StructuralHash this is identical to HashMultiMap.
    type HashMultiMap<'key,'a,'hashTag> when 'hashTag :> IEqualityComparer<'key> with 
        /// Create a new empty mutable hash table with an internal bucket array of the given approximate size
        /// and with the given key hash/equality functions
        static member Create: 'hashTag * int             -> HashMultiMap<'key,'a,'hashTag>
        /// Create a new empty mutable hash table with an internal bucket array of the given approximate size
        /// and with the given key hash/equality functions
        static member Create: 'hashTag * HashStats * int             -> HashMultiMap<'key,'a,'hashTag>
        /// Make a shallow copy of the collection
        member Copy    : unit    -> HashMultiMap<'key,'a,'hashTag>
        /// Add a binding for the element to the table
        member Add     : 'key * 'a -> unit
        /// Clear all elements from the collection
        member Clear   : unit    -> unit
        /// Test if the collection contains any bindings for the given element
        member Contains: 'key      -> bool
        /// Remove the latest binding (if any) for the given element from the table
        member Remove  : 'key      -> unit
        /// Replace the latest binding (if any) for the given element.
        member Replace : 'key * 'a -> unit

        /// Lookup or set the given element in the table.  Raise [[Not_found]]/[[KeyNotFoundException]] if the element is not found.
        member Item : 'key -> 'a with get,set

        [<System.Obsolete("Consider using hashTable.[key] instead")>]
        member Find    : 'key      -> 'a
        /// Lookup the given element in the table, returning the result as an Option
        member TryFind : 'key      -> 'a option
        /// Find all bindings for the given element in the table, if any
        member FindAll : 'key      -> 'a list

        /// Apply the given function to each element in the collection threading the accumulating parameter
        /// through the sequence of function applications
        member Fold    : ('key -> 'a -> 'c -> 'c) -> 'c -> 'c

        /// The number of bindings in the hash table
        member Count   : int

        /// Apply the given function to each binding in the hash table 
        member Iterate : ('key -> 'a -> unit) -> unit

        /// Get the longest chain in the hash table as a list.  Use for debugging
        /// poorly performing hash functions
        [<Obsolete("GetLongestChain is deprecated. It will return the empty list")>]
        member GetLongestChain : unit -> ('key * 'a) list 

    type HashMultiMap<'key,'a> = HashMultiMap<'key,'a, IEqualityComparer<'key>>    
    type [<System.Obsolete("This type has been renamed. Consider using Tagged.HashMultiMap<'key,'a> instead") >]
         HashMultiMapUntagged<'key,'a> = HashMultiMap<'key,'a>


    /// Mutable hash sets based on F# structural "hash" and (=) functions. Implemented via a hash table and/or Dictionary.
    type HashSet<'a> = HashSet<'a, IEqualityComparer<'a>>    
    and  [<System.Obsolete("This type has been renamed. Consider using Tagged.HashSet<'a> instead") >]
         HashSetUntagged<'a> = HashSet<'a>
    
    /// Mutable hash sets where a constraint tag tracks information about the hash/equality functions used
    /// for the hashing. When the tag is Tags.StructuralHash this is identical to HashSet.
    and HashSet<'a,'hashTag> when 'hashTag :> IEqualityComparer<'a>  with  
        /// Create a new empty mutable hash set with an internal bucket array of the given approximate size
        /// and with the given key hash/equality functions 
        static member Create: 'hashTag * int             -> HashSet<'a,'hashTag>

        /// Create a new empty mutable hash set with an internal bucket array of the given approximate size
        /// and with the given key hash/equality functions and the given HashStats collector
        [<Obsolete("HashStats are no longer supported. The HashStats argument will be ignored")>]
        static member Create: 'hashTag * HashStats * int -> HashSet<'a,'hashTag>
        
        /// Make a shallow copy of the set
        member Copy    : unit -> HashSet<'a,'hashTag>
        /// Add an element to the collection
        member Add     : 'a   -> unit
        /// Clear all elements from the set
        member Clear   : unit -> unit
        /// Test if the set contains the given element
        member Contains: 'a   -> bool
        /// Remove the given element from the set
        member Remove  : 'a   -> unit
        /// Apply the given function to the set threading the accumulating parameter
        /// through the sequence of function applications
        member Fold    : ('a -> 'b -> 'b) -> 'b -> 'b
        
        /// The number of elements in the set
        member Count   : int

        /// Apply the given function to each binding in the hash table 
        member Iterate : ('a -> unit) -> unit

        /// Use to debug poorly performing hash tables
        [<Obsolete("GetLongestChain is no longer supported. It will return the empty list")>]
        member GetLongestChain : unit -> 'a list 

#if CLI_AT_MOST_1_1
#else
        interface IEnumerable<'a> 
        interface System.Collections.IEnumerable 
#endif

    /// Immutable sets based on binary trees, where comparison is the
    /// F# structural comparison function.
    ///
    /// See the Set module for further operations on sets.
    ///
    /// These sets can be used with elements of any type, but you should check that
    /// structural hashing and equality on the element type are correct for your type.  
    ///
    /// Efficiency: Structural comparison is relatively efficient but is not a suitable
    /// choice in all circumstances, e.g. it may not compare efficiently on 
    /// non-reference types and deeply-structured types. 
    type Set<'a> = Set<'a, IComparer<'a>>    
    and [<System.Obsolete("This type has been renamed. Consider using Tagged.Set<'a> instead") >]
        SetUntagged<'a> = Set<'a>

    /// Immutable sets where a constraint tag carries information about the key-comparer 
    /// being used.  If 'comparerTag is Tags.StructuralComparer then 
    /// this type is the same as Set (the tag can be ignored).
    and Set<'a,'comparerTag> when 'comparerTag :> IComparer<'a> with 

        /// A useful shortcut for Set.add.  Note this operation prodcues a new set
        /// and does not mutate the original set.  The new set will share many storage
        /// nodes with the original.  See the Set module for further operations on sets.
        member Add : 'a -> Set<'a,'comparerTag>
        
        /// A useful shortcut for Set.remove.  Note this operation prodcues a new set
        /// and does not mutate the original set.  The new set will share many storage
        /// nodes with the original.  See the Set module for further operations on sets.
        member Remove : 'a -> Set<'a,'comparerTag>
        
        /// A useful shortcut for Set.size.  See the Set module for further operations on sets.
        [<System.Obsolete("Consider using the .Count property instead")>]
        member Size : int

        /// Return the number of elements in the set
        member Count : int
        
        /// A useful shortcut for Set.mem.  See the Set module for further operations on sets.
        member Contains : 'a -> bool
        
        /// A useful shortcut for Set.is_empty.  See the Set module for further operations on sets.
        member IsEmpty  : bool

        /// Apply the given function to each binding in the collection
        member Iterate : ('a -> unit) -> unit

        /// Apply the given accumulating function to all the elements of the set
        member Fold    : ('a -> 'b -> 'b) -> 'b -> 'b

        /// Build two new sets, one containing the elements for which the given predicate returns 'true',
        /// and the other the remaining elements.
        member Partition: ('a -> bool) -> Set<'a,'comparerTag> * Set<'a,'comparerTag>

        /// Return a new collection containing only the elements of the collection
        /// for which the given predicate returns "true"
        member Filter: ('a -> bool) -> Set<'a,'comparerTag> 

        /// Test if any element of the collection satisfies the given predicate.
        /// If the elements are "i0...iN" then computes "p i0 or ... or p iN".
        member Exists: ('a -> bool) -> bool

        /// Test if all elements of the collection satisfy the given predicate.
        /// If the elements are "i0...iN" and "j0...jN" then computes "p i0 &amp;&amp; ... &amp;&amp; p iN".
        member ForAll: ('a -> bool) -> bool

        /// The empty set based on the given comparer
        static member Create: 'comparerTag * seq<'a> -> Set<'a,'comparerTag> 
        
        /// The empty set based on the given comparer
        static member Empty: 'comparerTag -> Set<'a,'comparerTag> 
        
        /// A singleton set based on the given comparison operator
        static member Singleton: 'comparerTag * 'a -> Set<'a,'comparerTag> 
        
        /// Compares two sets and returns true if they are equal or false otherwise
        static member Equality : Set<'a,'comparerTag> * Set<'a,'comparerTag> -> bool
        
        /// Compares a and b and returns 1 if a &gt; b, -1 if b &lt; a and 0 if a = b        
        static member Compare : a:Set<'a,'comparerTag> * b:Set<'a,'comparerTag> -> int

        /// Return a new set with the elements of the second set removed from the first.
        static member (-) : Set<'a,'comparerTag> * Set<'a,'comparerTag> -> Set<'a,'comparerTag> 

        /// Compute the union of the two sets.
        static member (+) : Set<'a,'comparerTag> * Set<'a,'comparerTag> -> Set<'a,'comparerTag> 

        /// Compute the intersection of the two sets.
        static member Intersection : Set<'a,'comparerTag> * Set<'a,'comparerTag> -> Set<'a,'comparerTag> 

        /// Compute the union of the two sets.
        static member Union : Set<'a,'comparerTag> * Set<'a,'comparerTag> -> Set<'a,'comparerTag>

        /// Return a new set with the elements of the second set removed from the first.
        static member Difference: Set<'a,'comparerTag> * Set<'a,'comparerTag> -> Set<'a,'comparerTag> 

        /// The number of elements in the set
        member Choose : 'a 

        /// Returns the lowest element in the set according to the ordering being used for the set
        member MinimumElement: 'a

        /// Returns the highest element in the set according to the ordering being used for the set
        member MaximumElement: 'a

        /// Evaluates to "true" if all elements of the second set are in the first
        member IsSubsetOf: Set<'a,'comparerTag> -> bool

        /// Evaluates to "true" if all elements of the first set are in the second
        member IsSupersetOf: Set<'a,'comparerTag> -> bool

        /// The elements of the set as a list.
        member ToList : unit -> 'a list
        
        /// The elements of the set as an array.
        member ToArray: unit -> 'a array 

#if CLI_AT_MOST_1_1
#else
        interface ICollection<'a> 
#endif
        interface IEnumerable<'a> 
        interface System.Collections.IEnumerable 

        interface System.IComparable
        override Equals : obj -> bool

        //interface IStructuralHash

        [<Obsolete("This should always return true")>]
        member CheckBalanceInvariant : bool


    /// Immutable maps.  Keys are ordered by construction function specified
    /// when creating empty maps or by F# structural comparison if no
    /// construction function is specified.
    ///
    /// <performance> 
    ///   Maps based on structural comparison are  
    ///   efficient for small keys. They are not a suitable choice if keys are recursive data structures 
    ///   or require non-structural comparison semantics.
    /// </performance>
    type Map<'key,'a> = Map<'key, 'a, IComparer<'key>>    
    and  [<System.Obsolete("This type has been renamed. Consider using Tagged.Map<'key,'a> instead") >]
         MapUntagged<'key,'a> = Map<'key,'a>

    /// Immutable maps.  A constraint tag carries information about the key-comparer 
    /// being used.  If 'comparerTag is Tags.StructuralComparer then 
    /// this type is the same as Set (the tag can be ignored).
    and  Map<'key,'a,'comparerTag>  when 'comparerTag :> IComparer<'key> with 
        /// Return a new map with the binding added to the given map.
        member Add: 'key * 'a -> Map<'key,'a,'comparerTag>

        /// Return true if there are no bindings in the map.
        member IsEmpty: bool

        /// The empty map, and use the given comparer comparison function for all operations associated
        /// with any maps built from this map.
        static member Empty: 'comparerTag -> Map<'key,'a,'comparerTag>

        static member FromList : 'comparerTag * ('key * 'a) list -> Map<'key,'a,'comparerTag>

        /// Build a map that contains the bindings of the given IEnumerable
        /// and where comparison of elements is based on the given comparison function
        static member Create: 'comparerTag * seq<'key * 'a> -> Map<'key,'a,'comparerTag> 

        /// Test is an element is in the domain of the map
        member ContainsKey: 'key -> bool

        /// The number of bindings in the map
        member Count: int

        [<System.Obsolete("Consider using map.[key] instead")>]
        member Find: 'key -> 'a

        /// Lookup an element in the map, raising [[Not_found]]/[[KeyNotFoundException]] if no binding
        /// exists in the map.
        member Item : 'key -> 'a with get

        /// Search the map looking for the first element where the given function returns a [[Some]] value
        member First: ('key -> 'a -> 'b option) -> 'b option

        /// Return true if the given predicate returns true for all of the
        /// bindings in the map. Always returns true if the map is empty.
        member ForAll: ('key -> 'a -> bool) -> bool

        /// Return true if the given predicate returns true for one of the
        /// bindings in the map. Always returns false if the map is empty.
        member Exists: ('key -> 'a -> bool) -> bool

        /// Build a new map containing the bindings for which the given predicate returns 'true'.
        member Filter: ('key -> 'a -> bool) -> Map<'key,'a,'comparerTag> 

        /// Fold over the bindings in the map.  
        member Fold: ('key -> 'a -> 'state -> 'state) -> 'state -> 'state

        /// Given the start and end points of a key range,
        /// Fold over the bindings in the map that are in the range,
        /// and the end points are included if present (the range is considered a closed interval).
        member FoldSection: 'key -> 'key -> ('key -> 'a -> 'state -> 'state) -> 'state -> 'state

        /// Fold over the bindings in the map.  
        member FoldAndMap: ('key -> 'a -> 'state -> 'b * 'state) -> 'state -> Map<'key,'b,'comparerTag> * 'state

        /// Apply the given function to each binding in the dictionary
        member Iterate: ('key -> 'a -> unit) -> unit

        /// Build a new collection whose elements are the results of applying the given function
        /// to each of the elements of the collection. The index passed to the
        /// function indicates the index of element being transformed.
        member Map: ('key -> 'a -> 'b) -> Map<'key,'b,'comparerTag>

        /// Build a new collection whose elements are the results of applying the given function
        /// to each of the elements of the collection.
        member MapRange: ('a -> 'b) -> Map<'key,'b,'comparerTag>

        /// Build two new maps, one containing the bindings for which the given predicate returns 'true',
        /// and the other the remaining bindings.
        member Partition: ('key -> 'a -> bool) -> Map<'key,'a,'comparerTag> * Map<'key,'a,'comparerTag>

        /// Remove an element from the domain of the map.  No exception is raised if the element is not present.
        member Remove: 'key -> Map<'key,'a,'comparerTag>

        /// Lookup an element in the map, returning a [[Some]] value if the element is in the domain 
        /// of the map and [[None]] if not.
        member TryFind: 'key -> 'a option

        /// The elements of the set as a list.
        member ToList : unit -> ('key * 'a) list
    
        /// The elements of the set as an array
        member ToArray: unit -> ('key * 'a) array 

        interface IEnumerable<KeyValuePair<'key, 'a>>         
        
        interface System.Collections.IEnumerable 
        interface System.IComparable
        override Equals : obj -> bool

  
    
namespace Microsoft.FSharp.Collections
    open Microsoft.FSharp.Core
    open Microsoft.FSharp.Collections
    open System
#if CLI_AT_MOST_1_1
    open Microsoft.FSharp.Compatibility
#else
    open System.Collections.Generic
#endif

#if CLI_AT_MOST_1_1
#else          
    module CommonExtensions =

        val (|KeyValue|) : KeyValuePair<'a,'b> -> 'a * 'b
#endif

