// (c) Microsoft Corporation. All rights reserved 

#light

namespace Microsoft.FSharp.Math

    open Microsoft.FSharp.Collections
    open Microsoft.FSharp.Core
    open Microsoft.FSharp.Core.Operators
    open Microsoft.FSharp.Core.LanguagePrimitives.IntrinsicOperators
    open Microsoft.FSharp.Primitives.Basics
    open Microsoft.FSharp.Math
    open Microsoft.FSharp.Math.Primitives
    open System
    open System.Globalization
    module N = BigNat   

    // INVARIANT: sign = 1 or -1
    //            value(z) = sign * v
    // NOTE: 0 has two repns (+1,0) or (-1,0).
    [<Struct>]
    type BigInt(sign:int, v : BigNat) =

        //new() = new BigInt(0,0)

        member internal x.Sign = sign
        member internal x.V = v

        static member Equals (x:BigInt, y:BigInt) =
            match x.Sign,y.Sign with
            |  1, 1 -> N.equal x.V y.V                    // +1.xv = +1.yv iff xv = yv 
            | -1, -1 -> N.equal x.V y.V                    // -1.xv = -1.yv iff xv = yv 
            |  1,-1 -> N.isZero x.V && N.isZero y.V       //  1.xv = -1.yv iff xv=0 and yv=0 
            | -1, 1 -> N.isZero x.V && N.isZero y.V       // -1.xv =  1.yv iff xv=0 and yv=0 
            | _ -> invalid_arg "z signs should be +/- 1"
                
        static member LessThan(x:BigInt, y:BigInt) =
            match x.Sign,y.Sign with
            |  1, 1 -> N.lt x.V y.V                       //  1.xv <  1.yv iff xv < yv 
            | -1,-1 -> N.lt y.V x.V                       // -1.xv < -1.yv iff yv < xv 
            |  1,-1 -> false                              //  1.xv < -1.yv iff 0 <= 1.xv < -1.yv <= 0 iff false 
            | -1, 1 -> not (N.isZero x.V) || not (N.isZero y.V)
                                                          // -1.xv <  1.yv
                                   // (a) xv=0 and yv=0,  then false
                                   // (b) xv<>0,          -1.xv <  0 <= 1.yv, so true
                                   // (c) yv<>0,          -1.xv <= 0 <  1.yv, so true
            | _ -> invalid_arg "z signs should be +/- 1"
                
        static member GreaterThan (x:BigInt, y:BigInt) = // Follow lt by +/- symmetry 
            match x.Sign,y.Sign with
            | 1, 1 -> N.gt x.V y.V
            | -1,-1 -> N.gt y.V x.V
            |  1,-1 -> not (N.isZero x.V) || not (N.isZero y.V)
            | -1, 1 -> false
            | _ -> invalid_arg "z signs should be +/- 1"

        static member Compare(n,nn) = if BigInt.LessThan(n,nn) then -1 elif BigInt.Equals(n,nn) then 0 else 1
        static member Hash (z:BigInt) = z.Sign + N.hash(z.V)

        override x.ToString() =
            match x.Sign with
            |  1 -> N.to_string x.V                       // positive 
            | -1 -> 
                if N.isZero x.V             
                then "0"                    // not negative infact, but zero. 
                else "-" + N.to_string x.V  // negative 
            | _ -> invalid_arg "z signs should be +/- 1"
                

        interface System.IComparable with 
            member this.CompareTo(that:obj) = BigInt.Compare(this,(that :?> BigInt))
        override this.Equals(that) = BigInt.Equals(this, (that :?> BigInt))
  
        interface IStructuralHash with 
            member x.GetStructuralHashCode(nodesRemaining) = BigInt.Hash(x) // ignore nodesRemaining
        override x.GetHashCode() = BigInt.Hash(x)

    [<CompilationRepresentation(CompilationRepresentationFlags.ModuleSuffix)>]
    module BigInt = 

        type z = BigInt 

        let smallLim =  4096
        let smallPosTab = Array.init smallLim (fun i -> BigInt(1, N.of_int i))
        let smallNegTab = Array.init smallLim (fun i -> BigInt(-1, N.of_int i))
        let create s n = 
            if N.is_small n && N.get_small n < smallLim then (if s = 1 then smallPosTab else smallNegTab).[N.get_small n]
            else BigInt(s, n)
        let posn n = create 1 n
        let negn n = create (-1) n

        let of_int n = 
            if n>=0 
            then posn (N.of_int   n)
            elif (n = System.Int32.MinValue) then negn (N.of_int64 (-(int64 n)))
            else negn (N.of_int (-n))

        let to_int (z:z) = z.Sign * N.to_int z.V 
        let to_int32 (z:z) = to_int z
        let of_int32 (x:int32) = of_int x

        let of_int64 n = 
            if n>=0L 
            then posn (N.of_int64   n)
            elif (n = System.Int64.MinValue) then negn (N.add (N.of_int64 System.Int64.MaxValue) N.one) 
            else negn (N.of_int64 (-n))

        let to_int64 (z:z) = int64 z.Sign * N.to_int64 z.V 

        let hash (z:z) = BigInt.Hash(z)

        let one  = of_int 1
        let zero = of_int 0
        let neg (z:z)  = create (-1 * z.Sign) z.V
        let scale k (z:z) =
            if k<0
            then create (-z.Sign) (N.scale (-k) z.V)  // k.zsign.zv = -zsign.(-k.zv) 
            else create z.Sign    (N.scale k z.V)     // k.zsign.zv =  zsign.k.zv 

        // Result: 1.nx - 1.ny  (integer subtraction) 
        let subnn nx ny =                         
            if N.gte nx ny 
            then posn (N.sub nx ny)          // nx >= ny, result +ve,  +1.(nx - ny) 
            else negn (N.sub ny nx)          // nx < ny,  result -ve,  -1.(ny - nx) 

        let addnn nx ny = posn (N.add nx ny)              // Compute "nx + ny" to be integer 
            
        let isZero (x:z) = N.isZero x.V                   // signx.xv = 0 iff xv=0, since signx is +1,-1 
        let isOne  (x:z) = (x.Sign = 1) && N.isOne x.V       // signx.xv = 1 iff signx = +1 and xv = 1 
        let add x y =
            if isZero y then x else
            if isZero x then y else
            match x.Sign,y.Sign with
            |  1, 1 -> addnn x.V y.V                //  1.xv +  1.yv =  (xv + yv) 
            | -1,-1 -> neg (addnn x.V y.V)          // -1.xv + -1.yv = -(xv + yv) 
            |  1,-1 -> subnn x.V y.V                //  1.xv + -1.yv =  (xv - yv) 
            | -1, 1 -> subnn y.V x.V                // -1.xv +  1.yv =  (yv - xv) 
            | _ -> invalid_arg "z signs should be +/- 1"
                
        let sub (x:z) y =
            if isZero y then x else
            match x.Sign,y.Sign with
            |  1, 1 -> subnn x.V y.V                //  1.xv -  1.yv =  (xv - yv) 
            | -1,-1 -> subnn y.V x.V                // -1.xv - -1.yv =  (yv - xv) 
            |  1,-1 -> addnn x.V y.V                //  1.xv - -1.yv =  (xv + yv) 
            | -1, 1 -> neg (addnn x.V y.V)          // -1.xv -  1.yv = -(xv + yv) 
            | _ -> invalid_arg "z signs should be +/- 1"
                
        let mul x y = 
            if isZero x then x
            elif isZero y then y
            elif isOne x then y
            elif isOne y then x
            else 
                let m = (N.mul x.V y.V)
//                sample "smallMulResult" (N.is_small m) smallMulResult
                create (x.Sign * y.Sign) m  // xsign.xv * ysign.yv = (xsign.ysign).(xv.yv) 
            
        let divmod (x:z) (y:z) =
            let d,r = N.divmod x.V y.V 
            // HAVE: |x| = d.|y| + r and 0 <= r < |y| 
            // HAVE: xv  = d.yv  + r and 0 <= r < yv  
            match x.Sign,y.Sign with
            |  1, 1 -> posn d,posn r                //  1.xv =  1.d.( 1.yv) + ( 1.r) 
            | -1,-1 -> posn d,negn r                // -1.xv =  1.d.(-1.yv) + (-1.r) 
            |  1,-1 -> negn d,posn r                //  1.xv = -1.d.(-1.yv) + ( 1.r) 
            | -1, 1 -> negn d,negn r                // -1.xv = -1.d.( 1.yv) + (-1.r) 
            | _ -> invalid_arg "z signs should be +/- 1"
                
        let div x y = fst (divmod x y)
        let rem x y = snd (divmod x y)
            
        let hcf (x:z) (y:z) = posn (N.hcf x.V y.V) // hcf (xsign.xv,ysign.yv) = hcf (xv,yv) 
            
        let min (x:z) (y:z) =
            match x.Sign,y.Sign with
            |  1, 1 -> if N.lte x.V y.V then x else y   // if xv <= yv then  1.xv <=  1.yv 
            | -1,-1 -> if N.lte x.V y.V then y else x   // if xv <= yv then -1.yv <= -1.xv 
            |  1,-1 -> y                                // -1.yv <= 1.xv 
            | -1, 1 -> x                                // -1.xv <= 1.yv 
            | _ -> invalid_arg "z signs should be +/- 1"
                
        let max (x:z) (y:z) =
            match x.Sign,y.Sign with
            |  1, 1 -> if N.gte x.V y.V then x else y   // if xv >= yv then  1.xv >=  1.yv 
            | -1,-1 -> if N.gte x.V y.V then y else x   // if xv >= yv then -1.yv >= -1.xv 
            |  1,-1 -> x                                //  1.xv >= -1.yv 
            | -1, 1 -> y                                //  1.yv >= -1.xv 
            | _ -> invalid_arg "z signs should be +/- 1"
                
        let negative (x:z) = x.Sign = -1 && not (isZero x)  // signx.xv < 0 iff signx = -1 and xv<>0 
        let positive (x:z) = x.Sign =  1 && not (isZero x)  // signx.xv > 0 iff signx = +1 and xv<>0 
            
        let sign (x:z)  = x.Sign
        let abs (x:z)  = if x.Sign = -1 then neg x else x
          
        let equal (x:z) (y:z) = BigInt.Equals(x,y)
        let lt (x:z) (y:z) = BigInt.LessThan(x,y)
        let gt (x:z) (y:z) = BigInt.GreaterThan(x,y)

        let lte (x:z) (y:z) =
            match x.Sign,y.Sign with
            |  1, 1 -> N.lte x.V y.V                      //  1.xv <=  1.yv iff xv <= yv 
            | -1,-1 -> N.lte y.V x.V                      // -1.xv <= -1.yv iff yv <= xv 
            |  1,-1 -> N.isZero x.V && N.isZero y.V       //  1.xv <= -1.yv,
                                                          // (a) if xv=0 and yv=0 then true
                                                          // (b) otherwise false, only meet at zero.
                                                           
            | -1, 1 -> true                               // -1.xv <= 1.yv, true 
            | _ -> invalid_arg "z signs should be +/- 1"
                
        let gte (x:z) (y:z) = // Follow lte by +/- symmetry 
            match x.Sign,y.Sign with
            |  1, 1 -> N.gte x.V y.V
            | -1,-1 -> N.gte y.V x.V
            |  1,-1 -> true
            | -1, 1 -> N.isZero x.V && N.isZero y.V
            | _ -> invalid_arg "z signs should be +/- 1"
                
        let powi (z:z) i =
            if i>=0 then
             // (signz.zv)^i = (signz^i)        .(zv^i)
             //              = (signz^(i mod 2)).(zv^i) since sign is 1,-1
             //              = either     1.zv^i        when i mod 2 = 0 (even power kills sign)
             //                    or signz.zv^i        when i mod 2 = 1 (odd  power keeps sign)
                 
                create 
                  (if i % 2 = 0 then 1 else z.Sign)
                  (N.powi z.V i)
            else
                invalid_arg "powi x i: requires i be non -ve"
              
        let pow (z:z) i =
            if not (negative i) then
                create (if N.isZero (N.rem i.V N.two) then 1 else z.Sign)
                       (N.pow z.V i.V)
            else
                invalid_arg "pow x i: requires i be non -ve"
              
        let to_float (x:z) =
            match x.Sign with
            |  1 ->    N.to_float x.V                     // float (1.xv)  =   float (xv) 
            | -1 -> - (N.to_float x.V)                    // float (-1.xv) = - float (xv) 
            | _ -> invalid_arg "z signs should be +/- 1"
                
        let to_string (x:z) = x.ToString()
        
        let of_string (str:string) =
            let len = str.Length 
            if len = 0 then invalid_arg "of_string: empty string";
            if str.[0..0] = "-" then
                negn (N.of_string str.[1..len-1])
            else
                posn (N.of_string str)
              
        let compare n nn = if lt n nn then -1 elif equal n nn then 0 else 1
          
        let bits (x:z) = N.bits (x.V)
        let is_small (x:z) = N.is_small (x.V)
        let factorial x =
            if lt x zero then failwith "factorial(x) had x<0" else
            posn (N.factorial x.V)


    type BigInt with 
        static member ( + )(n1,n2) = BigInt.add n1 n2
        static member ( % )(n1,n2) = BigInt.rem n1 n2
        static member ( * )(n1,n2) = BigInt.mul n1 n2
        static member ( - )(n1,n2) = BigInt.sub n1 n2 
        static member ( / )(n1,n2) = BigInt.div n1 n2
        static member ( ~- )(n1)  = BigInt.neg n1
        static member ( ~+ )(n1:BigInt) = n1
        static member (..) (n1:BigInt,n2:BigInt) = StandardRanges.generate BigInt.zero BigInt.add n1 BigInt.one n2
        static member (.. ..) (n1:BigInt,step:BigInt,n2:BigInt) = StandardRanges.generate BigInt.zero BigInt.add n1 step n2
  
        static member ToDouble(x) = BigInt.to_float x
        static member FromInt64(x) = BigInt.of_int64 x
        static member FromInt32(x) = BigInt.of_int32 x
        static member ToInt64(x) = BigInt.to_int64 x
        static member ToInt32(x) = BigInt.to_int x
        static member Parse(s) = BigInt.of_string s
        member x.IsNegative = BigInt.negative x
        member x.IsPositive = BigInt.positive x
        member x.IsZero = BigInt.isZero x
        member x.IsOne = BigInt.isOne x

namespace Microsoft.FSharp.Core

    type bigint = Microsoft.FSharp.Math.BigInt

namespace Microsoft.FSharp.Math.Types

    type BigInt = Microsoft.FSharp.Math.BigInt
