/* PgSqlClient - ADO.NET Data Provider for PostgreSQL 7.4+
 * Copyright (c) 2003-2004 Carlos Guzman Alvarez
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

using System;
using System.Data;
using System.Drawing;
using System.Data.Common;
using System.ComponentModel;
using System.ComponentModel.Design;

namespace PostgreSql.Data.PgSqlClient
{	
	#region DELEGATES 

	public delegate void PgRowUpdatedEventHandler(object sender, PgRowUpdatedEventArgs e);
	public delegate void PgRowUpdatingEventHandler(object sender, PgRowUpdatingEventArgs e);

	#endregion

	[ToolboxItem(true),
	ToolboxBitmap(typeof(PgDataAdapter), "Resources.ToolBox.PgDataAdapter.bmp"),	
	DefaultEvent("RowUpdated")]
	public sealed class PgDataAdapter : DbDataAdapter, IDbDataAdapter
	{
		#region Events

		private static readonly object EventRowUpdated = new object(); 
		private static readonly object EventRowUpdating = new object(); 

		#endregion

		#region Fields

		private PgCommand	selectCommand;
		private PgCommand	insertCommand;
		private PgCommand	updateCommand;
		private PgCommand	deleteCommand;
		private bool		disposed;

		#endregion

		#region Properties

		IDbCommand IDbDataAdapter.SelectCommand 
		{
			get { return selectCommand; }
			set { selectCommand = (PgCommand)value; }
		}

		IDbCommand IDbDataAdapter.InsertCommand 
		{
			get { return insertCommand; }
			set { insertCommand = (PgCommand)value; }
		}

		IDbCommand IDbDataAdapter.UpdateCommand 
		{
			get { return updateCommand; }
			set { updateCommand = (PgCommand)value; }
		}

		IDbCommand IDbDataAdapter.DeleteCommand 
		{
			get { return deleteCommand; }
			set { deleteCommand = (PgCommand)value; }
		}

		[Category("DataCategory_Update"), DefaultValue(null)]
		public PgCommand SelectCommand 
		{
			get { return selectCommand; }
			set { selectCommand = value; }
		}

		[Category("DataCategory_Update"), DefaultValue(null)]
		public PgCommand InsertCommand 
		{
			get { return insertCommand; }
			set { insertCommand = value; }
		}
		
		[Category("DataCategory_Fill"), DefaultValue(null)]
		public PgCommand UpdateCommand 
		{
			get { return updateCommand; }
			set { updateCommand = value; }
		}

		[Category("DataCategory_Update"), DefaultValue(null)]
		public PgCommand DeleteCommand 
		{
			get { return deleteCommand; }
			set { deleteCommand = value; }
		}

		#endregion

		#region Constructors

		public PgDataAdapter() : base()
		{
			GC.SuppressFinalize(this);
		}

		public PgDataAdapter(PgCommand selectCommand) : this()
		{
			this.selectCommand = selectCommand;
		}
		
		public PgDataAdapter(string commandText, PgConnection connection) : this()
		{
			this.selectCommand = new PgCommand(commandText, connection);
		}

		public PgDataAdapter(string commandText, string connectionString) : this()
		{
			this.selectCommand = new PgCommand(commandText, new PgConnection(connectionString));
		}

		#endregion

		#region IDisposable Methods

		protected override void Dispose(bool disposing)
		{
			if (!disposed)
			{
				try
				{
					if (disposing)
					{
						if (selectCommand != null)
						{
							selectCommand.Dispose();
						}
						if (insertCommand != null)
						{
							insertCommand.Dispose();
						}
						if (updateCommand != null)
						{
							updateCommand.Dispose();
						}
					}
					
					// release any unmanaged resources
					
					disposed = true;
				}
				finally 
				{
					base.Dispose(disposing);
				}
			}
		}

		#endregion

		#region Methods

		public event PgRowUpdatedEventHandler RowUpdated
		{
			add { Events.AddHandler(EventRowUpdated, value); }
			remove { Events.RemoveHandler(EventRowUpdated, value); }
		}

		[Category("DataCategory_Update")]
		public event PgRowUpdatingEventHandler RowUpdating
		{
			add { Events.AddHandler(EventRowUpdating, value); }
			remove { Events.RemoveHandler(EventRowUpdating, value); }
		}

		#endregion

		#region Protected Methods
		
		protected override RowUpdatedEventArgs CreateRowUpdatedEvent(DataRow dataRow, IDbCommand command, StatementType statementType, DataTableMapping tableMapping)
		{
			return new PgRowUpdatedEventArgs(dataRow, command, statementType, tableMapping);
		}

		protected override void OnRowUpdated(RowUpdatedEventArgs value)
		{
			PgRowUpdatedEventHandler handler = (PgRowUpdatedEventHandler) Events[EventRowUpdated];
			if ((null != handler) && (value is PgRowUpdatedEventArgs)) 
			{
				handler(this, (PgRowUpdatedEventArgs) value);
			}
		}

		protected override RowUpdatingEventArgs CreateRowUpdatingEvent(DataRow dataRow, IDbCommand command, StatementType statementType, DataTableMapping tableMapping)
		{
			return new PgRowUpdatingEventArgs(dataRow, command, statementType, tableMapping);
		}

		protected override void OnRowUpdating(RowUpdatingEventArgs value)
		{
			PgRowUpdatingEventHandler handler = (PgRowUpdatingEventHandler) Events[EventRowUpdating];
			if ((null != handler) && (value is PgRowUpdatingEventArgs)) 
			{
				handler(this, (PgRowUpdatingEventArgs) value);
			}
		}

		#endregion
	}
}
