/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.camel.kotlin.components

import kotlin.Boolean
import kotlin.String
import kotlin.Unit
import org.apache.camel.kotlin.CamelDslMarker
import org.apache.camel.kotlin.UriDsl

/**
 * Detect and parse documents using Workday.
 */
public fun UriDsl.workday(i: WorkdayUriDsl.() -> Unit) {
  WorkdayUriDsl(this).apply(i)
}

@CamelDslMarker
public class WorkdayUriDsl(
  it: UriDsl,
) {
  private val it: UriDsl

  init {
    this.it = it
    this.it.component("workday")
  }

  private var entity: String = ""

  private var path: String = ""

  /**
   * The entity to be requested or subscribed via API.
   */
  public fun entity(entity: String) {
    this.entity = entity
    it.url("$entity:$path")
  }

  /**
   * The API path to access an entity structure.
   */
  public fun path(path: String) {
    this.path = path
    it.url("$entity:$path")
  }

  /**
   * Whether the producer should be started lazy (on the first message). By starting lazy you can
   * use this to allow CamelContext and routes to startup in situations where a producer may otherwise
   * fail during starting and cause the route to fail being started. By deferring this startup to be
   * lazy then the startup failure can be handled during routing messages via Camel's routing error
   * handlers. Beware that when the first message is processed then creating and starting the producer
   * may take a little time and prolong the total processing time of the processing.
   */
  public fun lazyStartProducer(lazyStartProducer: String) {
    it.property("lazyStartProducer", lazyStartProducer)
  }

  /**
   * Whether the producer should be started lazy (on the first message). By starting lazy you can
   * use this to allow CamelContext and routes to startup in situations where a producer may otherwise
   * fail during starting and cause the route to fail being started. By deferring this startup to be
   * lazy then the startup failure can be handled during routing messages via Camel's routing error
   * handlers. Beware that when the first message is processed then creating and starting the producer
   * may take a little time and prolong the total processing time of the processing.
   */
  public fun lazyStartProducer(lazyStartProducer: Boolean) {
    it.property("lazyStartProducer", lazyStartProducer.toString())
  }

  /**
   * Pool connection manager for advanced configuration.
   */
  public fun httpConnectionManager(httpConnectionManager: String) {
    it.property("httpConnectionManager", httpConnectionManager)
  }

  /**
   * Workday Report as a service output format.
   */
  public fun reportFormat(reportFormat: String) {
    it.property("reportFormat", reportFormat)
  }

  /**
   * Workday Host name.
   */
  public fun host(host: String) {
    it.property("host", host)
  }

  /**
   * Workday client Id generated by API client for integrations.
   */
  public fun clientId(clientId: String) {
    it.property("clientId", clientId)
  }

  /**
   * Workday client Secret generated by API client for integrations.
   */
  public fun clientSecret(clientSecret: String) {
    it.property("clientSecret", clientSecret)
  }

  /**
   * Workday token Refresh generated for integrations system user.
   */
  public fun tokenRefresh(tokenRefresh: String) {
    it.property("tokenRefresh", tokenRefresh)
  }

  /**
   * Workday Tenant name.
   */
  public fun tenant(tenant: String) {
    it.property("tenant", tenant)
  }
}
