
import static com.google.flatbuffers.Constants.*;
import static com.google.common.truth.Truth.assertThat;

import DictionaryLookup.*;
import MyGame.Example.*;
import com.google.common.io.ByteStreams;
import optional_scalars.ScalarStuff;
import optional_scalars.OptionalByte;
import NamespaceA.*;
import NamespaceA.NamespaceB.*;
import com.google.flatbuffers.ByteBufferUtil;
import com.google.flatbuffers.ByteVector;
import com.google.flatbuffers.FlatBufferBuilder;
import com.google.flatbuffers.FlexBuffers;
import com.google.flatbuffers.FlexBuffersBuilder;
import com.google.flatbuffers.StringVector;
import com.google.flatbuffers.UnionVector;

import com.google.flatbuffers.FlexBuffers.FlexBufferException;
import com.google.flatbuffers.FlexBuffers.Reference;
import com.google.flatbuffers.FlexBuffers.Vector;
import com.google.flatbuffers.ArrayReadWriteBuf;
import com.google.flatbuffers.FlexBuffers.KeyVector;

import java.io.*;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.channels.FileChannel;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import org.junit.Rule;
import org.junit.rules.TemporaryFolder;
import org.junit.runner.RunWith;
import org.junit.runners.JUnit4;


/*
 * Copyright 2014 Google Inc. All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

@RunWith(JUnit4.class)
public class JavaTest {

    @Rule
    public TemporaryFolder tempFolder = new TemporaryFolder();

    @org.junit.Test
    public void mainTest() throws IOException {
      // First, let's test reading a FlatBuffer generated by C++ code:
      // This file was generated from monsterdata_test.json
      byte[] data = ByteStreams.toByteArray(
        JavaTest.class.getClassLoader().getResourceAsStream("monsterdata_test.mon"));

      // Now test it:
      ByteBuffer bb = ByteBuffer.wrap(data);
      TestBuffer(bb);
      TestPackUnpack(bb);
    }

    @org.junit.Test
    public void testFlatBufferBuilder() {
      // We use an initial size of 1 to exercise the reallocation algorithm,
      // normally a size larger than the typical FlatBuffer you generate would be
      // better for performance.
      FlatBufferBuilder fbb = new FlatBufferBuilder(1);
      TestBuilderBasics(fbb, true);
      TestBuilderBasics(fbb, false);
      TestExtendedBuffer(fbb.dataBuffer().asReadOnlyBuffer());
    }

    @org.junit.Test
    public void TestEnums() {
      assertThat(Color.name(Color.Red)).isEqualTo("Red");
      assertThat(Color.name(Color.Blue)).isEqualTo("Blue");
      assertThat(Any.name(Any.NONE)).isEqualTo("NONE");
      assertThat(Any.name(Any.Monster)).isEqualTo("Monster");
    }

    static void TestBuffer(ByteBuffer bb) {
      assertThat(Monster.MonsterBufferHasIdentifier(bb)).isEqualTo(true);

      Monster monster = Monster.getRootAsMonster(bb);

      assertThat(monster.hp()).isEqualTo((short) 80);
      // default
      assertThat(monster.mana()).isEqualTo((short) 150);

      assertThat(monster.name()).isEqualTo("MyMonster");
      // monster.friendly() // can't access, deprecated

        Vec3 pos = monster.pos();
      assertThat(pos.x()).isEqualTo(1.0f);
      assertThat(pos.y()).isEqualTo(2.0f);
      assertThat(pos.z()).isEqualTo(3.0f);
      assertThat(pos.test1()).isEqualTo(3.0);
      // issue: int != byte
      assertThat(pos.test2()).isEqualTo((int) Color.Green);
      Test t = pos.test3();
      assertThat(t.a()).isEqualTo((short) 5);
      assertThat(t.b()).isEqualTo((byte) 6);

      assertThat(monster.testType()).isEqualTo((byte) Any.Monster);
      Monster monster2 = new Monster();
      assertThat(monster.test(monster2) != null).isTrue();
      assertThat(monster2.name()).isEqualTo("Fred");

      assertThat(monster.inventoryLength()).isEqualTo(5);
      int invsum = 0;
        for (int i = 0; i < monster.inventoryLength(); i++)
            invsum += monster.inventory(i);
      assertThat(invsum).isEqualTo(10);

      // Method using a vector access object:
        ByteVector inventoryVector = monster.inventoryVector();
      assertThat(inventoryVector.length()).isEqualTo(5);
      invsum = 0;
        for (int i = 0; i < inventoryVector.length(); i++)
            invsum += inventoryVector.getAsUnsigned(i);
      assertThat(invsum).isEqualTo(10);

      // Alternative way of accessing a vector:
        ByteBuffer ibb = monster.inventoryAsByteBuffer();
        invsum = 0;
        while (ibb.position() < ibb.limit())
            invsum += ibb.get();
      assertThat(invsum).isEqualTo(10);

      Test test_0 = monster.test4(0);
        Test test_1 = monster.test4(1);
      assertThat(monster.test4Length()).isEqualTo(2);
      assertThat(test_0.a() + test_0.b() + test_1.a() + test_1.b()).isEqualTo((Integer) 100);

      Test.Vector test4Vector = monster.test4Vector();
        test_0 = test4Vector.get(0);
        test_1 = test4Vector.get(1);
      assertThat(test4Vector.length()).isEqualTo(2);
      assertThat(test_0.a() + test_0.b() + test_1.a() + test_1.b()).isEqualTo((Integer) 100);

      assertThat(monster.testarrayofstringLength()).isEqualTo(2);
      assertThat(monster.testarrayofstring(0)).isEqualTo("test1");
      assertThat(monster.testarrayofstring(1)).isEqualTo("test2");

      // Method using a vector access object:
        StringVector testarrayofstringVector = monster.testarrayofstringVector();
      assertThat(testarrayofstringVector.length()).isEqualTo(2);
      assertThat(testarrayofstringVector.get(0)).isEqualTo("test1");
      assertThat(testarrayofstringVector.get(1)).isEqualTo("test2");

      assertThat(monster.testbool()).isEqualTo(true);
    }

    // this method checks additional fields not present in the binary buffer read from file
    // these new tests are performed on top of the regular tests
    static void TestExtendedBuffer(ByteBuffer bb) {
        TestBuffer(bb);

        Monster monster = Monster.getRootAsMonster(bb);

      assertThat(monster.testhashu32Fnv1()).isEqualTo((Integer.MAX_VALUE + 1L));
    }


    @org.junit.Test public void TestNamespaceNesting() {
        // reference / manipulate these to verify compilation
        FlatBufferBuilder fbb = new FlatBufferBuilder(1);

        TableInNestedNS.startTableInNestedNS(fbb);
        TableInNestedNS.addFoo(fbb, 1234);
        int nestedTableOff = TableInNestedNS.endTableInNestedNS(fbb);

        TableInFirstNS.startTableInFirstNS(fbb);
        TableInFirstNS.addFooTable(fbb, nestedTableOff);
        int off = TableInFirstNS.endTableInFirstNS(fbb);
    }

    @org.junit.Test public void TestNestedFlatBuffer() {
        final String nestedMonsterName = "NestedMonsterName";
        final short nestedMonsterHp = 600;
        final short nestedMonsterMana = 1024;

        FlatBufferBuilder fbb1 = new FlatBufferBuilder(16);
        int str1 = fbb1.createString(nestedMonsterName);
        Monster.startMonster(fbb1);
        Monster.addName(fbb1, str1);
        Monster.addHp(fbb1, nestedMonsterHp);
        Monster.addMana(fbb1, nestedMonsterMana);
        int monster1 = Monster.endMonster(fbb1);
        Monster.finishMonsterBuffer(fbb1, monster1);
        byte[] fbb1Bytes = fbb1.sizedByteArray();
        fbb1 = null;

        FlatBufferBuilder fbb2 = new FlatBufferBuilder(16);
        int str2 = fbb2.createString("My Monster");
        int nestedBuffer = Monster.createTestnestedflatbufferVector(fbb2, fbb1Bytes);
        Monster.startMonster(fbb2);
        Monster.addName(fbb2, str2);
        Monster.addHp(fbb2, (short)50);
        Monster.addMana(fbb2, (short)32);
        Monster.addTestnestedflatbuffer(fbb2, nestedBuffer);
        int monster = Monster.endMonster(fbb2);
        Monster.finishMonsterBuffer(fbb2, monster);

        // Now test the data extracted from the nested buffer
        Monster mons = Monster.getRootAsMonster(fbb2.dataBuffer());
        Monster nestedMonster = mons.testnestedflatbufferAsMonster();

      assertThat(nestedMonsterMana).isEqualTo(nestedMonster.mana());
      assertThat(nestedMonsterHp).isEqualTo(nestedMonster.hp());
      assertThat(nestedMonsterName).isEqualTo(nestedMonster.name());
    }

    @org.junit.Test public void TestCreateByteVector() {
        FlatBufferBuilder fbb = new FlatBufferBuilder(16);
        int str = fbb.createString("MyMonster");
        byte[] inventory = new byte[] { 0, 1, 2, 3, 4 };
        int vec = fbb.createByteVector(inventory);
        Monster.startMonster(fbb);
        Monster.addInventory(fbb, vec);
        Monster.addName(fbb, str);
        int monster1 = Monster.endMonster(fbb);
        Monster.finishMonsterBuffer(fbb, monster1);
        Monster monsterObject = Monster.getRootAsMonster(fbb.dataBuffer());

      assertThat((Integer) monsterObject.inventory(1)).isEqualTo((int)inventory[1]);
      assertThat(monsterObject.inventoryLength()).isEqualTo(inventory.length);
      ByteVector inventoryVector = monsterObject.inventoryVector();
      assertThat(inventoryVector.getAsUnsigned(1)).isEqualTo((int)inventory[1]);
      assertThat(inventoryVector.length()).isEqualTo(inventory.length);

      assertThat(ByteBuffer.wrap(inventory)).isEqualTo(
        monsterObject.inventoryAsByteBuffer());
    }

    @org.junit.Test public void TestCreateUninitializedVector() {
        FlatBufferBuilder fbb = new FlatBufferBuilder(16);
        int str = fbb.createString("MyMonster");
        byte[] inventory = new byte[] { 0, 1, 2, 3, 4 };
        ByteBuffer bb = fbb.createUnintializedVector(1, inventory.length, 1);
        for (byte i:inventory) {
            bb.put(i);
        }
        int vec = fbb.endVector();
        Monster.startMonster(fbb);
        Monster.addInventory(fbb, vec);
        Monster.addName(fbb, str);
        int monster1 = Monster.endMonster(fbb);
        Monster.finishMonsterBuffer(fbb, monster1);
        Monster monsterObject = Monster.getRootAsMonster(fbb.dataBuffer());

      assertThat((Integer) monsterObject.inventory(1)).isEqualTo((int)inventory[1]);
      assertThat(monsterObject.inventoryLength()).isEqualTo(inventory.length);
      ByteVector inventoryVector = monsterObject.inventoryVector();
      assertThat(inventoryVector.getAsUnsigned(1)).isEqualTo((int)inventory[1]);
      assertThat(inventoryVector.length()).isEqualTo(inventory.length);
      assertThat(ByteBuffer.wrap(inventory)).isEqualTo(
        monsterObject.inventoryAsByteBuffer());
    }

    @org.junit.Test public void TestByteBufferFactory() throws IOException {
      File file = tempFolder.newFile("javatest.bin");
        final class MappedByteBufferFactory extends FlatBufferBuilder.ByteBufferFactory {
            @Override
            public ByteBuffer newByteBuffer(int capacity) {
                ByteBuffer bb;
                try {
                    RandomAccessFile f = new RandomAccessFile(file, "rw");
                    bb =  f.getChannel().map(FileChannel.MapMode.READ_WRITE, 0, capacity).order(ByteOrder.LITTLE_ENDIAN);
                    f.close();
                } catch(Throwable e) {
                    System.out.println("FlatBuffers test: couldn't map ByteBuffer to a file");
                    bb = null;
                }
                return bb;
            }
        }

        FlatBufferBuilder fbb = new FlatBufferBuilder(1, new MappedByteBufferFactory());

        TestBuilderBasics(fbb, false);
    }

    @org.junit.Test public void TestSizedInputStream() {
        // Test on default FlatBufferBuilder that uses HeapByteBuffer
        FlatBufferBuilder fbb = new FlatBufferBuilder(1);

        TestBuilderBasics(fbb, false);

        InputStream in = fbb.sizedInputStream();
        byte[] array = fbb.sizedByteArray();
        int count = 0;
        int currentVal = 0;

        while (currentVal != -1 && count < array.length) {
            try {
                currentVal = in.read();
            } catch(java.io.IOException e) {
                System.out.println("FlatBuffers test: couldn't read from InputStream");
                return;
            }
          assertThat((byte)currentVal).isEqualTo(array[count]);
          count++;
        }
      assertThat(count).isEqualTo(array.length);
    }

    void TestBuilderBasics(FlatBufferBuilder fbb, boolean sizePrefix) {
        int[] names = {fbb.createString("Frodo"), fbb.createString("Barney"), fbb.createString("Wilma")};
        int[] off = new int[3];
        Monster.startMonster(fbb);
        Monster.addName(fbb, names[0]);
        off[0] = Monster.endMonster(fbb);
        Monster.startMonster(fbb);
        Monster.addName(fbb, names[1]);
        off[1] = Monster.endMonster(fbb);
        Monster.startMonster(fbb);
        Monster.addName(fbb, names[2]);
        off[2] = Monster.endMonster(fbb);
        int sortMons = fbb.createSortedVectorOfTables(new Monster(), off);

        // We set up the same values as monsterdata.json:

        int str = fbb.createString("MyMonster");

        int inv = Monster.createInventoryVector(fbb, new byte[] { 0, 1, 2, 3, 4 });

        int fred = fbb.createString("Fred");
        Monster.startMonster(fbb);
        Monster.addName(fbb, fred);
        int mon2 = Monster.endMonster(fbb);

        Monster.startTest4Vector(fbb, 2);
        Test.createTest(fbb, (short)10, (byte)20);
        Test.createTest(fbb, (short)30, (byte)40);
        int test4 = fbb.endVector();

        int testArrayOfString = Monster.createTestarrayofstringVector(fbb, new int[] {
                fbb.createString("test1"),
                fbb.createString("test2")
        });

        Monster.startMonster(fbb);
        Monster.addPos(fbb, Vec3.createVec3(fbb, 1.0f, 2.0f, 3.0f, 3.0,
                Color.Green, (short)5, (byte)6));
        Monster.addHp(fbb, (short)80);
        Monster.addName(fbb, str);
        Monster.addInventory(fbb, inv);
        Monster.addTestType(fbb, (byte)Any.Monster);
        Monster.addTest(fbb, mon2);
        Monster.addTest4(fbb, test4);
        Monster.addTestarrayofstring(fbb, testArrayOfString);
        Monster.addTestbool(fbb, true);
        Monster.addTesthashu32Fnv1(fbb, Integer.MAX_VALUE + 1L);
        Monster.addTestarrayoftables(fbb, sortMons);
        int mon = Monster.endMonster(fbb);

        if (sizePrefix) {
            Monster.finishSizePrefixedMonsterBuffer(fbb, mon);
        } else {
            Monster.finishMonsterBuffer(fbb, mon);
        }

        // Write the result to a file for debugging purposes:
        // Note that the binaries are not necessarily identical, since the JSON
        // parser may serialize in a slightly different order than the above
        // Java code. They are functionally equivalent though.

        try {
            String filename = "monsterdata_java_wire" + (sizePrefix ? "_sp" : "") + ".mon";
            FileChannel fc = new FileOutputStream(tempFolder.newFile(filename)).getChannel();
            fc.write(fbb.dataBuffer().duplicate());
            fc.close();
        } catch(java.io.IOException e) {
            System.out.println("FlatBuffers test: couldn't write file");
            return;
        }

        // Test it:
        ByteBuffer dataBuffer = fbb.dataBuffer();
        if (sizePrefix) {
          assertThat(ByteBufferUtil.getSizePrefix(dataBuffer) + SIZE_PREFIX_LENGTH).isEqualTo(
            dataBuffer.remaining());
          dataBuffer = ByteBufferUtil.removeSizePrefix(dataBuffer);
        }
        TestExtendedBuffer(dataBuffer);

        // Make sure it also works with read only ByteBuffers. This is slower,
        // since creating strings incurs an additional copy
        // (see Table.__string).
        TestExtendedBuffer(dataBuffer.asReadOnlyBuffer());

        //Attempt to mutate Monster fields and check whether the buffer has been mutated properly
        // revert to original values after testing
        Monster monster = Monster.getRootAsMonster(dataBuffer);

        // mana is optional and does not exist in the buffer so the mutation should fail
        // the mana field should retain its default value
      assertThat(monster.mutateMana((short)10)).isFalse();
      assertThat(monster.mana()).isEqualTo((short) 150);

      // Accessing a vector of sorted by the key tables
      assertThat(monster.testarrayoftables(0).name()).isEqualTo("Barney");
      assertThat(monster.testarrayoftables(1).name()).isEqualTo("Frodo");
      assertThat(monster.testarrayoftables(2).name()).isEqualTo("Wilma");
      Monster.Vector testarrayoftablesVector = monster.testarrayoftablesVector();
      assertThat(testarrayoftablesVector.get(0).name()).isEqualTo("Barney");
      assertThat(testarrayoftablesVector.get(1).name()).isEqualTo("Frodo");
      assertThat(testarrayoftablesVector.get(2).name()).isEqualTo("Wilma");

      // Example of searching for a table by the key
      assertThat(monster.testarrayoftablesByKey("Frodo").name()).isEqualTo("Frodo");
      assertThat(monster.testarrayoftablesByKey("Barney").name()).isEqualTo("Barney");
      assertThat(monster.testarrayoftablesByKey("Wilma").name()).isEqualTo("Wilma");
      assertThat(testarrayoftablesVector.getByKey("Frodo").name()).isEqualTo("Frodo");
      assertThat(testarrayoftablesVector.getByKey("Barney").name()).isEqualTo("Barney");
      assertThat(testarrayoftablesVector.getByKey("Wilma").name()).isEqualTo("Wilma");

      // testType is an existing field and mutating it should succeed
      assertThat(monster.testType()).isEqualTo((byte) Any.Monster);

      //mutate the inventory vector
      assertThat(monster.mutateInventory(0, 1)).isTrue();
      assertThat(monster.mutateInventory(1, 2)).isTrue();
      assertThat(monster.mutateInventory(2, 3)).isTrue();
      assertThat(monster.mutateInventory(3, 4)).isTrue();
      assertThat(monster.mutateInventory(4, 5)).isTrue();

      for (int i = 0; i < monster.inventoryLength(); i++) {
        assertThat((Integer) monster.inventory(i)).isEqualTo(i + 1);
      }
        ByteVector inventoryVector =  monster.inventoryVector();
        for (int i = 0; i < inventoryVector.length(); i++) {
          assertThat((int)inventoryVector.get(i)).isEqualTo(i + 1);
        }

        //reverse mutation
      assertThat(monster.mutateInventory(0, 0)).isTrue();
      assertThat(monster.mutateInventory(1, 1)).isTrue();
      assertThat(monster.mutateInventory(2, 2)).isTrue();
      assertThat(monster.mutateInventory(3, 3)).isTrue();
      assertThat(monster.mutateInventory(4, 4)).isTrue();

      // get a struct field and edit one of its fields
      Vec3 pos = monster.pos();
      assertThat(pos.x()).isEqualTo(1.0f);
      pos.mutateX(55.0f);
      assertThat(pos.x()).isEqualTo(55.0f);
      pos.mutateX(1.0f);
      assertThat(pos.x()).isEqualTo(1.0f);
    }

    @org.junit.Test public void TestVectorOfUnions() {
        final FlatBufferBuilder fbb = new FlatBufferBuilder();

        final int swordAttackDamage = 1;

        final int[] characterVector = new int[] {
            Attacker.createAttacker(fbb, swordAttackDamage),
        };

        final byte[] characterTypeVector = new byte[]{
            Character.MuLan,
        };

        Movie.finishMovieBuffer(
            fbb,
            Movie.createMovie(
                fbb,
                (byte)0,
                (byte)0,
                Movie.createCharactersTypeVector(fbb, characterTypeVector),
                Movie.createCharactersVector(fbb, characterVector)
            )
        );

        final Movie movie = Movie.getRootAsMovie(fbb.dataBuffer());
        ByteVector charactersTypeByteVector = movie.charactersTypeVector();
        UnionVector charactersVector = movie.charactersVector();

      assertThat(movie.charactersTypeLength()).isEqualTo(characterTypeVector.length);
      assertThat(charactersTypeByteVector.length()).isEqualTo(characterTypeVector.length);
      assertThat(movie.charactersLength()).isEqualTo(characterVector.length);
      assertThat(charactersVector.length()).isEqualTo(characterVector.length);

      assertThat((Byte) movie.charactersType(0)).isEqualTo(characterTypeVector[0]);
      assertThat(charactersTypeByteVector.get(0)).isEqualTo(characterTypeVector[0]);

      assertThat(((Attacker)movie.characters(new Attacker(), 0)).swordAttackDamage()).isEqualTo(
        swordAttackDamage);
    }

    @org.junit.Test public void TestFixedLengthArrays() {
        FlatBufferBuilder builder = new FlatBufferBuilder(0);

        float       a;
        int[]       b = new int[15];
        byte        c;
        int[][]     d_a = new int[2][2];
        byte[]      d_b = new byte[2];
        byte[][]    d_c = new byte[2][2];
        long[][]    d_d = new long[2][2];
        int         e;
        long[]      f = new long[2];

        a = 0.5f;
        for (int i = 0; i < 15; i++) b[i] = i;
        c = 1;
        d_a[0][0] = 1;
        d_a[0][1] = 2;
        d_a[1][0] = 3;
        d_a[1][1] = 4;
        d_b[0] = TestEnum.B;
        d_b[1] = TestEnum.C;
        d_c[0][0] = TestEnum.A;
        d_c[0][1] = TestEnum.B;
        d_c[1][0] = TestEnum.C;
        d_c[1][1] = TestEnum.B;
        d_d[0][0] = -1;
        d_d[0][1] = 1;
        d_d[1][0] = -2;
        d_d[1][1] = 2;
        e = 2;
        f[0] = -1;
        f[1] = 1;

        int arrayOffset = ArrayStruct.createArrayStruct(builder,
            a, b, c, d_a, d_b, d_c, d_d, e, f);

        // Create a table with the ArrayStruct.
        ArrayTable.startArrayTable(builder);
        ArrayTable.addA(builder, arrayOffset);
        int tableOffset = ArrayTable.endArrayTable(builder);

        ArrayTable.finishArrayTableBuffer(builder, tableOffset);

        ArrayTable table = ArrayTable.getRootAsArrayTable(builder.dataBuffer());
        NestedStruct nested = new NestedStruct();

      assertThat(table.a().a()).isEqualTo(0.5f);
      for (int i = 0; i < 15; i++)
        assertThat(table.a().b(i)).isEqualTo(i);
      assertThat(table.a().c()).isEqualTo((byte)1);
      assertThat(table.a().d(nested, 0).a(0)).isEqualTo(1);
      assertThat(table.a().d(nested, 0).a(1)).isEqualTo(2);
      assertThat(table.a().d(nested, 1).a(0)).isEqualTo(3);
      assertThat(table.a().d(nested, 1).a(1)).isEqualTo(4);
      assertThat(table.a().d(nested, 0).b()).isEqualTo(TestEnum.B);
      assertThat(table.a().d(nested, 1).b()).isEqualTo(TestEnum.C);
      assertThat(table.a().d(nested, 0).c(0)).isEqualTo(TestEnum.A);
      assertThat(table.a().d(nested, 0).c(1)).isEqualTo(TestEnum.B);
      assertThat(table.a().d(nested, 1).c(0)).isEqualTo(TestEnum.C);
      assertThat(table.a().d(nested, 1).c(1)).isEqualTo(TestEnum.B);
      assertThat(table.a().d(nested, 0).d(0)).isEqualTo((long)-1);
      assertThat(table.a().d(nested, 0).d(1)).isEqualTo((long)1);
      assertThat(table.a().d(nested, 1).d(0)).isEqualTo((long)-2);
      assertThat(table.a().d(nested, 1).d(1)).isEqualTo((long)2);
      assertThat(table.a().e()).isEqualTo(2);
      assertThat(table.a().f(0)).isEqualTo((long)-1);
      assertThat(table.a().f(1)).isEqualTo((long)1);
    }

    @org.junit.Test public void testFlexBuffersTest() {
        FlexBuffersBuilder builder = new FlexBuffersBuilder(ByteBuffer.allocate(512),
                FlexBuffersBuilder.BUILDER_FLAG_SHARE_KEYS_AND_STRINGS);
        testFlexBuffersTest(builder);
        int bufferLimit1 = ((ArrayReadWriteBuf) builder.getBuffer()).limit();

        // Repeat after clearing the builder to ensure the builder is reusable
        builder.clear();
        testFlexBuffersTest(builder);
        int bufferLimit2 = ((ArrayReadWriteBuf) builder.getBuffer()).limit();
      assertThat(bufferLimit1).isEqualTo(bufferLimit2);
    }

    public static void testFlexBuffersTest(FlexBuffersBuilder builder) {
        // Write the equivalent of:
        // { vec: [ -100, "Fred", 4.0, false ], bar: [ 1, 2, 3 ], bar3: [ 1, 2, 3 ],
        // foo: 100, bool: true, mymap: { foo: "Fred" } }
        // It's possible to do this without std::function support as well.
        int map1 = builder.startMap();

        int vec1 = builder.startVector();
        builder.putInt(-100);
        builder.putString("Fred");
        builder.putBlob(new byte[]{(byte) 77});
        builder.putBoolean(false);
        builder.putInt(Long.MAX_VALUE);

        int map2 = builder.startMap();
        builder.putInt("test", 200);
        builder.endMap(null, map2);

        builder.putFloat(150.9);
        builder.putFloat(150.9999998);
        builder.endVector("vec", vec1, false, false);

        vec1 = builder.startVector();
        builder.putInt(1);
        builder.putInt(2);
        builder.putInt(3);
        builder.endVector("bar", vec1, true, false);

        vec1 = builder.startVector();
        builder.putBoolean(true);
        builder.putBoolean(false);
        builder.putBoolean(true);
        builder.putBoolean(false);
        builder.endVector("bools", vec1, true, false);

        builder.putBoolean("bool", true);
        builder.putFloat("foo", 100);

        map2 = builder.startMap();
        builder.putString("bar", "Fred");  // Testing key and string reuse.
        builder.putInt("int", -120);
        builder.putFloat("float", -123.0f);
        builder.putBlob("blob", new byte[]{ 65, 67 });
        builder.endMap("mymap", map2);

        builder.endMap(null, map1);
        builder.finish();

        FlexBuffers.Map m = FlexBuffers.getRoot(builder.getBuffer()).asMap();

      assertThat(m.size()).isEqualTo(6);

      // test empty (an null)
      // empty if fail
      assertThat(m.get("no_key").asString()).isEqualTo("");
      // empty if fail
      assertThat(m.get("no_key").asMap()).isEqualTo(FlexBuffers.Map.empty());
      // empty if fail
      assertThat(m.get("no_key").asKey()).isEqualTo(FlexBuffers.Key.empty());
      // empty if fail
      assertThat(m.get("no_key").asVector()).isEqualTo(Vector.empty());
      // empty if fail
      assertThat(m.get("no_key").asBlob()).isEqualTo(FlexBuffers.Blob.empty());
      assert(m.get("no_key").asVector().isEmpty()); // empty if fail

        // testing "vec" field
        FlexBuffers.Vector vec = m.get("vec").asVector();
      assertThat(vec.size()).isEqualTo(8);
      assertThat(vec.get(0).asLong()).isEqualTo((long) -100);
      assertThat(vec.get(1).asString()).isEqualTo("Fred");
      assertThat(vec.get(2).isBlob()).isTrue();
      assertThat(vec.get(2).asBlob().size()).isEqualTo(1);
      assertThat(vec.get(2).asBlob().data().get(0)).isEqualTo((byte) 77);
      // Check if type is a bool
      assertThat(vec.get(3).isBoolean()).isTrue();
      // Check if value is false
      assertThat(vec.get(3).asBoolean()).isFalse();
      assertThat(vec.get(4).asLong()).isEqualTo(Long.MAX_VALUE);
      assertThat(vec.get(5).isMap()).isTrue();
      assertThat(vec.get(5).asMap().get("test").asInt()).isEqualTo(200);
      assertThat(Float.compare((float)vec.get(6).asFloat(), 150.9f)).isEqualTo(0);
      assertThat(Double.compare(vec.get(7).asFloat(), 150.9999998)).isEqualTo(0);
      //conversion fail returns 0 as C++
      assertThat((long)0).isEqualTo((long)vec.get(1).asLong());

      // bar vector
        FlexBuffers.Vector tvec = m.get("bar").asVector();
      assertThat(tvec.size()).isEqualTo(3);
      assertThat(tvec.get(0).asInt()).isEqualTo(1);
      assertThat(tvec.get(1).asInt()).isEqualTo(2);
      assertThat(tvec.get(2).asInt()).isEqualTo(3);
      assertThat(((FlexBuffers.TypedVector) tvec).getElemType()).isEqualTo(FlexBuffers.FBT_INT);

      // bools vector
        FlexBuffers.Vector bvec = m.get("bools").asVector();
      assertThat(bvec.size()).isEqualTo(4);
      assertThat(bvec.get(0).asBoolean()).isTrue();
      assertThat(bvec.get(1).asBoolean()).isFalse();
      assertThat(bvec.get(2).asBoolean()).isTrue();
      assertThat(bvec.get(3).asBoolean()).isFalse();
      assertThat(((FlexBuffers.TypedVector) bvec).getElemType()).isEqualTo(FlexBuffers.FBT_BOOL);

      assertThat((float)m.get("foo").asFloat()).isEqualTo((float) 100);
      assertThat(m.get("unknown").isNull()).isTrue();

      // mymap vector
        FlexBuffers.Map mymap = m.get("mymap").asMap();
      // These should be equal by pointer equality, since key and value are shared.
      assertThat(mymap.keys().get(0)).isEqualTo(m.keys().get(0));
      assertThat(mymap.keys().get(0).toString()).isEqualTo("bar");
      assertThat(mymap.values().get(0).asString()).isEqualTo(vec.get(1).asString());
      assertThat(mymap.get("int").asInt()).isEqualTo(-120);
      assertThat((float)mymap.get("float").asFloat()).isEqualTo(-123.0f);
      assertThat(Arrays.equals(mymap.get("blob").asBlob().getBytes(), new byte[]{ 65, 67 })).isEqualTo(
        true);
      assertThat(mymap.get("blob").asBlob().toString()).isEqualTo("AC");
      assertThat(mymap.get("blob").toString()).isEqualTo("\"AC\"");
    }

    @org.junit.Test public void testFlexBufferVectorStrings() {
        FlexBuffersBuilder builder = new FlexBuffersBuilder(ByteBuffer.allocate(10000000));

        int size = 3000;
        StringBuilder sb = new StringBuilder();
        for (int i=0; i< size; i++) {
            sb.append("a");
        }

        String text = sb.toString();
      assertThat(text.length()).isEqualTo(size);

      int pos = builder.startVector();

        for (int i=0; i<size; i++) {
            builder.putString(text);
        }

        try {
            builder.endVector(null, pos, true, false);
            // this should raise an exception as
            // typed vector of string was deprecated
            assert false;
        } catch(FlexBufferException fb) {
            // no op
        }
        // we finish the vector again as non-typed
        builder.endVector(null, pos, false, false);

        ByteBuffer b = builder.finish();
        Vector v = FlexBuffers.getRoot(b).asVector();

      assertThat(v.size()).isEqualTo(size);
      for (int i=0; i<size; i++) {
        assertThat(v.get(i).asString().length()).isEqualTo(size);
        assertThat(v.get(i).asString()).isEqualTo(text);
      }
    }

    @org.junit.Test public void testDeprecatedTypedVectorString() {
        // tests whether we are able to support reading deprecated typed vector string
        // data is equivalent to [ "abc", "abc", "abc", "abc"]
        byte[] data = new byte[] {0x03, 0x61, 0x62, 0x63, 0x00, 0x03, 0x61, 0x62, 0x63, 0x00,
            0x03, 0x61, 0x62, 0x63, 0x00, 0x03, 0x61, 0x62, 0x63, 0x00, 0x04, 0x14, 0x10,
             0x0c, 0x08, 0x04, 0x3c, 0x01};
        Reference ref = FlexBuffers.getRoot(ByteBuffer.wrap(data));
      assertThat(ref.getType()).isEqualTo(FlexBuffers.FBT_VECTOR_STRING_DEPRECATED);
      assertThat(ref.isTypedVector()).isTrue();
      Vector vec = ref.asVector();
        for (int i=0; i< vec.size(); i++) {
          assertThat("abc").isEqualTo(vec.get(i).asString());
        }
    }

    @org.junit.Test public void testSingleElementBoolean() {
        FlexBuffersBuilder builder = new FlexBuffersBuilder(ByteBuffer.allocate(100));
        builder.putBoolean(true);
        ByteBuffer b = builder.finish();
        assertThat(FlexBuffers.getRoot(b).asBoolean()).isTrue();
    }

    @org.junit.Test public void testSingleElementByte() {
        FlexBuffersBuilder builder = new FlexBuffersBuilder();
        builder.putInt(10);
        ByteBuffer b = builder.finish();
      assertThat(10).isEqualTo(FlexBuffers.getRoot(b).asInt());
    }

    @org.junit.Test public void testSingleElementShort() {
        FlexBuffersBuilder builder = new FlexBuffersBuilder();
        builder.putInt(Short.MAX_VALUE);
        ByteBuffer b = builder.finish();
      assertThat(Short.MAX_VALUE).isEqualTo((short) FlexBuffers.getRoot(b).asInt());
    }

    @org.junit.Test public void testSingleElementInt() {
        FlexBuffersBuilder builder = new FlexBuffersBuilder();
        builder.putInt(Integer.MIN_VALUE);
        ByteBuffer b = builder.finish();
      assertThat(Integer.MIN_VALUE).isEqualTo(FlexBuffers.getRoot(b).asInt());
    }

    @org.junit.Test public void testSingleElementLong() {
        FlexBuffersBuilder builder = new FlexBuffersBuilder();
        builder.putInt(Long.MAX_VALUE);
        ByteBuffer b = builder.finish();
      assertThat(Long.MAX_VALUE).isEqualTo(FlexBuffers.getRoot(b).asLong());
    }

    @org.junit.Test public void testSingleElementFloat() {
        FlexBuffersBuilder builder = new FlexBuffersBuilder();
        builder.putFloat(Float.MAX_VALUE);
        ByteBuffer b = builder.finish();
      assertThat(Float.compare(Float.MAX_VALUE, (float) FlexBuffers.getRoot(b).asFloat())).isEqualTo(
        0);
    }

    @org.junit.Test public void testSingleElementDouble() {
        FlexBuffersBuilder builder = new FlexBuffersBuilder();
        builder.putFloat(Double.MAX_VALUE);
        ByteBuffer b = builder.finish();
      assertThat(Double.compare(Double.MAX_VALUE, FlexBuffers.getRoot(b).asFloat())).isEqualTo(0);
    }

    @org.junit.Test public void testSingleElementBigString() {
        FlexBuffersBuilder builder = new FlexBuffersBuilder(ByteBuffer.allocate(10000));
        StringBuilder sb = new StringBuilder();

        for (int i=0; i< 3000; i++) {
            sb.append("a");
        }

        builder.putString(sb.toString());
        ByteBuffer b = builder.finish();

        FlexBuffers.Reference r = FlexBuffers.getRoot(b);

      assertThat(FlexBuffers.FBT_STRING).isEqualTo(r.getType());
      assertThat(sb.toString()).isEqualTo(r.asString());
    }

    @org.junit.Test public void testSingleElementSmallString() {
        FlexBuffersBuilder builder = new FlexBuffersBuilder(ByteBuffer.allocate(10000));

        builder.putString("aa");
        ByteBuffer b = builder.finish();
        FlexBuffers.Reference r = FlexBuffers.getRoot(b);

      assertThat(FlexBuffers.FBT_STRING).isEqualTo(r.getType());
      assertThat("aa").isEqualTo(r.asString());
    }

    @org.junit.Test public void testSingleElementBlob() {
        FlexBuffersBuilder builder = new FlexBuffersBuilder();
        builder.putBlob(new byte[]{5, 124, 118, -1});
        ByteBuffer b = builder.finish();
        FlexBuffers.Reference r = FlexBuffers.getRoot(b);
        byte[] result = r.asBlob().getBytes();
      assertThat((byte)5).isEqualTo(result[0]);
      assertThat((byte)124).isEqualTo(result[1]);
      assertThat((byte)118).isEqualTo(result[2]);
      assertThat((byte)-1).isEqualTo(result[3]);
    }

    @org.junit.Test public void testSingleElementLongBlob() {

        // verifies blobs of up to 2^16 in length
        for (int i = 2; i <= 1<<16; i = i<<1) {
            byte[] input = new byte[i-1];
            for (int index = 0; index < input.length; index++) {
                input[index] = (byte)(index % 64);
            }

            FlexBuffersBuilder builder = new FlexBuffersBuilder();
            builder.putBlob(input);
            ByteBuffer b = builder.finish();
            FlexBuffers.Reference r = FlexBuffers.getRoot(b);
            byte[] result = r.asBlob().getBytes();
            
            for (int index = 0; index < input.length; index++) {
              assertThat((byte)(index % 64)).isEqualTo(result[index]);
            }
        }
    }

    @org.junit.Test public void testSingleElementUByte() {
        FlexBuffersBuilder builder = new FlexBuffersBuilder();
        builder.putUInt(0xFF);
        ByteBuffer b = builder.finish();
        FlexBuffers.Reference r = FlexBuffers.getRoot(b);
      assertThat(255).isEqualTo((int)r.asUInt());
    }

    @org.junit.Test public void testSingleElementUShort() {
        FlexBuffersBuilder builder = new FlexBuffersBuilder();
        builder.putUInt(0xFFFF);
        ByteBuffer b = builder.finish();
        FlexBuffers.Reference r = FlexBuffers.getRoot(b);
      assertThat(65535).isEqualTo((int)r.asUInt());
    }

    @org.junit.Test public void testSingleElementUInt() {
        FlexBuffersBuilder builder = new FlexBuffersBuilder();
        builder.putUInt(0xFFFF_FFFFL);
        ByteBuffer b = builder.finish();
        FlexBuffers.Reference r = FlexBuffers.getRoot(b);
      assertThat(4294967295L).isEqualTo(r.asUInt());
    }

    @org.junit.Test public void testSingleFixedTypeVector() {

        int[] ints = new int[]{5, 124, 118, -1};
        float[] floats = new float[]{5.5f, 124.124f, 118.118f, -1.1f};
        String[] strings = new String[]{"This", "is", "a", "typed", "array"};
        boolean[] booleans = new boolean[]{false, true, true, false};


        FlexBuffersBuilder builder = new FlexBuffersBuilder(ByteBuffer.allocate(512),
                FlexBuffersBuilder.BUILDER_FLAG_NONE);

        int mapPos = builder.startMap();

        int vecPos = builder.startVector();
        for (final int i : ints) {
            builder.putInt(i);
        }
        builder.endVector("ints", vecPos, true, false);

        vecPos = builder.startVector();
        for (final float i : floats) {
            builder.putFloat(i);
        }
        builder.endVector("floats", vecPos, true, false);

        vecPos = builder.startVector();
        for (final boolean i : booleans) {
            builder.putBoolean(i);
        }
        builder.endVector("booleans", vecPos, true, false);

        builder.endMap(null, mapPos);


        ByteBuffer b = builder.finish();
        FlexBuffers.Reference r = FlexBuffers.getRoot(b);
        assert(r.asMap().get("ints").isTypedVector());
        assert(r.asMap().get("floats").isTypedVector());
        assert(r.asMap().get("booleans").isTypedVector());
    }

    @org.junit.Test public void testSingleElementVector() {
        FlexBuffersBuilder b = new FlexBuffersBuilder();

        int vecPos = b.startVector();
        b.putInt(99);
        b.putString("wow");
        b.putNull();
        int vecpos2 = b.startVector();
        b.putInt(99);
        b.putString("wow");
        b.putNull();
        b.endVector(null, vecpos2, false, false);
        b.endVector(null, vecPos, false, false);
        b.finish();

        FlexBuffers.Reference r = FlexBuffers.getRoot(b.getBuffer());
      assertThat(FlexBuffers.FBT_VECTOR).isEqualTo(r.getType());
      FlexBuffers.Vector vec = FlexBuffers.getRoot(b.getBuffer()).asVector();
      assertThat(4).isEqualTo(vec.size());
      assertThat(99).isEqualTo(vec.get(0).asInt());
      assertThat("wow").isEqualTo(vec.get(1).asString());
      assertThat(true).isEqualTo(vec.get(2).isNull());
      assertThat("[ 99, \"wow\", null ]").isEqualTo(vec.get(3).toString());
      assertThat("[ 99, \"wow\", null, [ 99, \"wow\", null ] ]").isEqualTo(
        FlexBuffers.getRoot(b.getBuffer()).toString());
    }

    @org.junit.Test public void testSingleElementMap() {
        FlexBuffersBuilder b = new FlexBuffersBuilder();

        int mapPost = b.startMap();
        b.putInt("myInt", 0x7fffffbbbfffffffL);
        b.putString("myString", "wow");
        b.putString("myString2", "incredible");
        b.putNull("myNull");
        int start = b.startVector();
        b.putInt(99);
        b.putString("wow");
        b.endVector("myVec", start, false, false);

        b.putFloat("double", 0x1.ffffbbbffffffP+1023);
        b.endMap(null, mapPost);
        b.finish();

        FlexBuffers.Reference r = FlexBuffers.getRoot(b.getBuffer());
      assertThat(FlexBuffers.FBT_MAP).isEqualTo(r.getType());
      FlexBuffers.Map map = FlexBuffers.getRoot(b.getBuffer()).asMap();
      assertThat(6).isEqualTo(map.size());
      assertThat(0x7fffffbbbfffffffL).isEqualTo(map.get("myInt").asLong());
      assertThat("wow").isEqualTo(map.get("myString").asString());
      assertThat("incredible").isEqualTo(map.get("myString2").asString());
      assertThat(true).isEqualTo(map.get("myNull").isNull());
      assertThat(99).isEqualTo(map.get("myVec").asVector().get(0).asInt());
      assertThat("wow").isEqualTo(map.get("myVec").asVector().get(1).asString());
      assertThat(Double.compare(0x1.ffffbbbffffffP+1023, map.get("double").asFloat())).isEqualTo(0);
      assertThat(
        "{ \"double\" : 1.7976894783391937E308, \"myInt\" : 9223371743723257855, \"myNull\" : null, \"myString\" : \"wow\", \"myString2\" : \"incredible\", \"myVec\" : [ 99, \"wow\" ] }").isEqualTo(
        FlexBuffers.getRoot(b.getBuffer()).toString());
    }

    @org.junit.Test public void testFlexBuferEmpty() {
        FlexBuffers.Blob blob = FlexBuffers.Blob.empty();
        FlexBuffers.Map ary = FlexBuffers.Map.empty();
        FlexBuffers.Vector map = FlexBuffers.Vector.empty();
        FlexBuffers.TypedVector typedAry = FlexBuffers.TypedVector.empty();
      assertThat(blob.size()).isEqualTo(0);
      assertThat(map.size()).isEqualTo(0);
      assertThat(ary.size()).isEqualTo(0);
      assertThat(typedAry.size()).isEqualTo(0);
    }

    @org.junit.Test public void testHashMapToMap() {
        int entriesCount = 12;

        HashMap<String, String> source =  new HashMap<>();
        for (int i = 0; i < entriesCount; i++) {
            source.put("foo_param_" + i, "foo_value_" + i);
        }

        FlexBuffersBuilder builder = new FlexBuffersBuilder(1000);
        int mapStart = builder.startMap();
        for (Map.Entry<String, String> entry : source.entrySet()) {
            builder.putString(entry.getKey(), entry.getValue());
        }
        builder.endMap(null, mapStart);
        ByteBuffer bb = builder.finish();
        bb.rewind();

        FlexBuffers.Reference rootReference = FlexBuffers.getRoot(bb);

      assertThat(rootReference.isMap()).isTrue();

      FlexBuffers.Map flexMap = rootReference.asMap();

        FlexBuffers.KeyVector keys = flexMap.keys();
        FlexBuffers.Vector values = flexMap.values();

      assertThat(entriesCount).isEqualTo(keys.size());
      assertThat(entriesCount).isEqualTo(values.size());

      HashMap<String, String> result =  new HashMap<>();
        for (int i = 0; i < keys.size(); i++) {
            result.put(keys.get(i).toString(), values.get(i).asString());
        }

      assertThat(source).isEqualTo(result);
    }

    @org.junit.Test public void testBuilderGrowth() {
        FlexBuffersBuilder builder = new FlexBuffersBuilder();
        String someString = "This is a small string";
        builder.putString(someString);
        ByteBuffer b = builder.finish();
      assertThat(someString).isEqualTo(FlexBuffers.getRoot(b).asString());

      FlexBuffersBuilder failBuilder = new FlexBuffersBuilder(ByteBuffer.allocate(1));
        failBuilder.putString(someString);
    }

    @org.junit.Test
    public void testFlexBuffersUtf8Map() {
        FlexBuffersBuilder builder = new FlexBuffersBuilder(ByteBuffer.allocate(512),
                FlexBuffersBuilder.BUILDER_FLAG_SHARE_KEYS_AND_STRINGS);

        String key0 = "😨 face1";
        String key1 = "😩 face2";
        String key2 = "😨 face3";
        String key3 = "trademark ®";
        String key4 = "€ euro";
        String utf8keys[] = { "😨 face1", "😩 face2", "😨 face3", "trademark ®", "€ euro"};

        int map = builder.startMap();

        for (int i=0; i< utf8keys.length; i++) {
            builder.putString(utf8keys[i], utf8keys[i]);  // Testing key and string reuse.
        }
        builder.endMap(null, map);
        builder.finish();

        FlexBuffers.Map m = FlexBuffers.getRoot(builder.getBuffer()).asMap();

      assertThat(m.size()).isEqualTo(5);

      KeyVector kv = m.keys();
        for (int i=0; i< utf8keys.length; i++) {
          assertThat(kv.get(i).toString()).isEqualTo(m.get(i).asString());
        }

      assertThat(m.get(key0).asString()).isEqualTo(utf8keys[0]);
      assertThat(m.get(key1).asString()).isEqualTo(utf8keys[1]);
      assertThat(m.get(key2).asString()).isEqualTo(utf8keys[2]);
      assertThat(m.get(key3).asString()).isEqualTo(utf8keys[3]);
      assertThat(m.get(key4).asString()).isEqualTo(utf8keys[4]);
    }

    @org.junit.Test public void testFlexBuffersMapLookup() {
        FlexBuffersBuilder builder = new FlexBuffersBuilder(ByteBuffer.allocate(512),
                FlexBuffersBuilder.BUILDER_FLAG_SHARE_KEYS_AND_STRINGS);

        String key0 = "123";
        String key1 = "1234";
        String key2 = "12345";
        String[] keys = new String[]{key0, key1, key2};

        int map = builder.startMap();

        for (int i=0; i< keys.length; i++) {
            builder.putString(keys[i], keys[i]);  // Testing key and string reuse.
        }
        builder.endMap(null, map);
        builder.finish();

        FlexBuffers.Map m = FlexBuffers.getRoot(builder.getBuffer()).asMap();
        for (int i=0; i< keys.length; i++) {
          assertThat(m.get(keys[i]).asString()).isEqualTo(keys[i]);
          assertThat(m.get(keys[i].getBytes(StandardCharsets.UTF_8)).asString()).isEqualTo(keys[i]);
        }
    }

    @org.junit.Test public void TestDictionaryLookup() {
        FlatBufferBuilder fbb = new FlatBufferBuilder(16);
        int lfIndex = LongFloatEntry.createLongFloatEntry(fbb, 0, 99);
        int vectorEntriesIdx = LongFloatMap.createEntriesVector(fbb, new int[] { lfIndex });
        int rootIdx = LongFloatMap.createLongFloatMap(fbb, vectorEntriesIdx);

        LongFloatMap.finishLongFloatMapBuffer(fbb, rootIdx);
        LongFloatMap map = LongFloatMap.getRootAsLongFloatMap(fbb.dataBuffer());
      assertThat(map.entriesLength()).isEqualTo(1);

      LongFloatEntry e = map.entries(0);
      assertThat(e.key()).isEqualTo(0L);
      assertThat(e.value()).isEqualTo(99.0f);

      LongFloatEntry e2 = map.entriesByKey(0);
      assertThat(e2.key()).isEqualTo(0L);
      assertThat(e2.value()).isEqualTo(99.0f);
    }

    @org.junit.Test public void TestVectorOfBytes() {
        FlatBufferBuilder fbb = new FlatBufferBuilder(16);
        int str = fbb.createString("ByteMonster");
        byte[] data = new byte[] {0, 1, 2, 3, 4, 5, 6, 7, 8, 9};
        int offset = Monster.createInventoryVector(fbb, data);
        Monster.startMonster(fbb);
        Monster.addName(fbb, str);
        Monster.addInventory(fbb, offset);
        int monster1 = Monster.endMonster(fbb);
        Monster.finishMonsterBuffer(fbb, monster1);
        Monster monsterObject = Monster.getRootAsMonster(fbb.dataBuffer());

      assertThat(monsterObject.inventoryLength()).isEqualTo(data.length);
      assertThat((Integer) monsterObject.inventory(4)).isEqualTo((int) data[4]);
      assertThat(ByteBuffer.wrap(data)).isEqualTo(monsterObject.inventoryAsByteBuffer());

      fbb.clear();
        ByteBuffer bb = ByteBuffer.wrap(data);
        offset = fbb.createByteVector(bb);
        str = fbb.createString("ByteMonster");
        Monster.startMonster(fbb);
        Monster.addName(fbb, str);
        Monster.addInventory(fbb, offset);
        monster1 = Monster.endMonster(fbb);
        Monster.finishMonsterBuffer(fbb, monster1);
        Monster monsterObject2 = Monster.getRootAsMonster(fbb.dataBuffer());

      assertThat(monsterObject2.inventoryLength()).isEqualTo(data.length);
      for (int i = 0; i < data.length; i++) {
        assertThat((Integer) monsterObject2.inventory(i)).isEqualTo((int) bb.get(i));
      }

        fbb.clear();
        offset = fbb.createByteVector(data, 3, 4);
        str = fbb.createString("ByteMonster");
        Monster.startMonster(fbb);
        Monster.addName(fbb, str);
        Monster.addInventory(fbb, offset);
        monster1 = Monster.endMonster(fbb);
        Monster.finishMonsterBuffer(fbb, monster1);
        Monster monsterObject3 = Monster.getRootAsMonster(fbb.dataBuffer());

      assertThat(monsterObject3.inventoryLength()).isEqualTo(4);
      assertThat((Integer) monsterObject3.inventory(0)).isEqualTo((int) data[3]);

      fbb.clear();
        bb = ByteBuffer.wrap(data);
        offset = Monster.createInventoryVector(fbb, bb);
        str = fbb.createString("ByteMonster");
        Monster.startMonster(fbb);
        Monster.addName(fbb, str);
        Monster.addInventory(fbb, offset);
        monster1 = Monster.endMonster(fbb);
        Monster.finishMonsterBuffer(fbb, monster1);
        Monster monsterObject4 = Monster.getRootAsMonster(fbb.dataBuffer());

      assertThat(monsterObject4.inventoryLength()).isEqualTo(data.length);
      assertThat((Integer) monsterObject4.inventory(8)).isEqualTo((int) 8);

      fbb.clear();
        byte[] largeData = new byte[1024];
        offset = fbb.createByteVector(largeData);
        str = fbb.createString("ByteMonster");
        Monster.startMonster(fbb);
        Monster.addName(fbb, str);
        Monster.addInventory(fbb, offset);
        monster1 = Monster.endMonster(fbb);
        Monster.finishMonsterBuffer(fbb, monster1);
        Monster monsterObject5 = Monster.getRootAsMonster(fbb.dataBuffer());

      assertThat(monsterObject5.inventoryLength()).isEqualTo(largeData.length);
      assertThat((Integer) monsterObject5.inventory(25)).isEqualTo((int) largeData[25]);

      fbb.clear();
        bb = ByteBuffer.wrap(largeData);
        bb.position(512);
        ByteBuffer bb2 = bb.slice();
      assertThat(bb2.arrayOffset()).isEqualTo(512);
      offset = fbb.createByteVector(bb2);
        str = fbb.createString("ByteMonster");
        Monster.startMonster(fbb);
        Monster.addName(fbb, str);
        Monster.addInventory(fbb, offset);
        monster1 = Monster.endMonster(fbb);
        Monster.finishMonsterBuffer(fbb, monster1);
        Monster monsterObject6 = Monster.getRootAsMonster(fbb.dataBuffer());

      assertThat(monsterObject6.inventoryLength()).isEqualTo(512);
      assertThat((Integer) monsterObject6.inventory(0)).isEqualTo((int) largeData[512]);

      fbb.clear();
        bb = ByteBuffer.wrap(largeData);
        bb.limit(256);
        offset = fbb.createByteVector(bb);
        str = fbb.createString("ByteMonster");
        Monster.startMonster(fbb);
        Monster.addName(fbb, str);
        Monster.addInventory(fbb, offset);
        monster1 = Monster.endMonster(fbb);
        Monster.finishMonsterBuffer(fbb, monster1);
        Monster monsterObject7 = Monster.getRootAsMonster(fbb.dataBuffer());

      assertThat(monsterObject7.inventoryLength()).isEqualTo(256);

      fbb.clear();
        bb = ByteBuffer.allocateDirect(2048);
        offset = fbb.createByteVector(bb);
        str = fbb.createString("ByteMonster");
        Monster.startMonster(fbb);
        Monster.addName(fbb, str);
        Monster.addInventory(fbb, offset);
        monster1 = Monster.endMonster(fbb);
        Monster.finishMonsterBuffer(fbb, monster1);
        Monster monsterObject8 = Monster.getRootAsMonster(fbb.dataBuffer());

      assertThat(monsterObject8.inventoryLength()).isEqualTo(2048);
    }

    @org.junit.Test public void TestSharedStringPool() {
        FlatBufferBuilder fb = new FlatBufferBuilder(1);
        String testString = "My string";
        int offset = fb.createSharedString(testString);
        for (int i=0; i< 10; i++) {
          assertThat(offset).isEqualTo(fb.createSharedString(testString));
        }
    }

    @org.junit.Test public void TestScalarOptional() {
        FlatBufferBuilder fbb = new FlatBufferBuilder(1);
        ScalarStuff.startScalarStuff(fbb);
        int pos = ScalarStuff.endScalarStuff(fbb);
        fbb.finish(pos);

        ScalarStuff scalarStuff = ScalarStuff.getRootAsScalarStuff(fbb.dataBuffer());
      assertThat(scalarStuff.justI8()).isEqualTo((byte)0);
      assertThat(scalarStuff.maybeI8()).isEqualTo((byte)0);
      assertThat(scalarStuff.defaultI8()).isEqualTo((byte)42);
      assertThat(scalarStuff.justU8()).isEqualTo(0);
      assertThat(scalarStuff.maybeU8()).isEqualTo(0);
      assertThat(scalarStuff.defaultU8()).isEqualTo(42);
      assertThat(scalarStuff.justI16()).isEqualTo((short)0);
      assertThat(scalarStuff.maybeI16()).isEqualTo((short)0);
      assertThat(scalarStuff.defaultI16()).isEqualTo((short)42);
      assertThat(scalarStuff.justU16()).isEqualTo(0);
      assertThat(scalarStuff.maybeU16()).isEqualTo(0);
      assertThat(scalarStuff.defaultU16()).isEqualTo(42);
      assertThat(scalarStuff.justI32()).isEqualTo(0);
      assertThat(scalarStuff.maybeI32()).isEqualTo(0);
      assertThat(scalarStuff.defaultI32()).isEqualTo(42);
      assertThat(scalarStuff.justU32()).isEqualTo(0L);
      assertThat(scalarStuff.maybeU32()).isEqualTo(0L);
      assertThat(scalarStuff.defaultU32()).isEqualTo(42L);
      assertThat(scalarStuff.justI64()).isEqualTo(0L);
      assertThat(scalarStuff.maybeI64()).isEqualTo(0L);
      assertThat(scalarStuff.defaultI64()).isEqualTo(42L);
      assertThat(scalarStuff.justU64()).isEqualTo(0L);
      assertThat(scalarStuff.maybeU64()).isEqualTo(0L);
      assertThat(scalarStuff.defaultU64()).isEqualTo(42L);
      assertThat(scalarStuff.justF32()).isEqualTo(0.0f);
      assertThat(scalarStuff.maybeF32()).isEqualTo(0f);
      assertThat(scalarStuff.defaultF32()).isEqualTo(42.0f);
      assertThat(scalarStuff.justF64()).isEqualTo(0.0);
      assertThat(scalarStuff.maybeF64()).isEqualTo(0.0);
      assertThat(scalarStuff.defaultF64()).isEqualTo(42.0);
      assertThat(scalarStuff.justBool()).isFalse();
      assertThat(scalarStuff.maybeBool()).isFalse();
      assertThat(scalarStuff.defaultBool()).isTrue();
      assertThat(scalarStuff.justEnum()).isEqualTo(OptionalByte.None);
      assertThat(scalarStuff.maybeEnum()).isEqualTo(OptionalByte.None);
      assertThat(scalarStuff.defaultEnum()).isEqualTo(OptionalByte.One);

      assertThat(scalarStuff.hasMaybeI8()).isFalse();
      assertThat(scalarStuff.hasMaybeI16()).isFalse();
      assertThat(scalarStuff.hasMaybeI32()).isFalse();
      assertThat(scalarStuff.hasMaybeI64()).isFalse();
      assertThat(scalarStuff.hasMaybeU8()).isFalse();
      assertThat(scalarStuff.hasMaybeU16()).isFalse();
      assertThat(scalarStuff.hasMaybeU32()).isFalse();
      assertThat(scalarStuff.hasMaybeU64()).isFalse();
      assertThat(scalarStuff.hasMaybeF32()).isFalse();
      assertThat(scalarStuff.hasMaybeF64()).isFalse();
      assertThat(scalarStuff.hasMaybeBool()).isFalse();
      assertThat(scalarStuff.hasMaybeEnum()).isFalse();

      fbb.clear();

        ScalarStuff.startScalarStuff(fbb);
        ScalarStuff.addJustI8(fbb, (byte)5);
        ScalarStuff.addMaybeI8(fbb, (byte)5);
        ScalarStuff.addDefaultI8(fbb, (byte)5);
        ScalarStuff.addJustU8(fbb, 6);
        ScalarStuff.addMaybeU8(fbb, 6);
        ScalarStuff.addDefaultU8(fbb, 6);
        ScalarStuff.addJustI16(fbb, (short)7);
        ScalarStuff.addMaybeI16(fbb, (short)7);
        ScalarStuff.addDefaultI16(fbb, (short)7);
        ScalarStuff.addJustU16(fbb, 8);
        ScalarStuff.addMaybeU16(fbb, 8);
        ScalarStuff.addDefaultU16(fbb, 8);
        ScalarStuff.addJustI32(fbb, 9);
        ScalarStuff.addMaybeI32(fbb, 9);
        ScalarStuff.addDefaultI32(fbb, 9);
        ScalarStuff.addJustU32(fbb, (long)10);
        ScalarStuff.addMaybeU32(fbb, (long)10);
        ScalarStuff.addDefaultU32(fbb, (long)10);
        ScalarStuff.addJustI64(fbb, 11L);
        ScalarStuff.addMaybeI64(fbb, 11L);
        ScalarStuff.addDefaultI64(fbb, 11L);
        ScalarStuff.addJustU64(fbb, 12L);
        ScalarStuff.addMaybeU64(fbb, 12L);
        ScalarStuff.addDefaultU64(fbb, 12L);
        ScalarStuff.addJustF32(fbb, 13.0f);
        ScalarStuff.addMaybeF32(fbb, 13.0f);
        ScalarStuff.addDefaultF32(fbb, 13.0f);
        ScalarStuff.addJustF64(fbb, 14.0);
        ScalarStuff.addMaybeF64(fbb, 14.0);
        ScalarStuff.addDefaultF64(fbb, 14.0);
        ScalarStuff.addJustBool(fbb, true);
        ScalarStuff.addMaybeBool(fbb, true);
        ScalarStuff.addDefaultBool(fbb, true);
        ScalarStuff.addJustEnum(fbb, OptionalByte.Two);
        ScalarStuff.addMaybeEnum(fbb, OptionalByte.Two);
        ScalarStuff.addDefaultEnum(fbb, OptionalByte.Two);

        pos = ScalarStuff.endScalarStuff(fbb);

        fbb.finish(pos);

        scalarStuff = ScalarStuff.getRootAsScalarStuff(fbb.dataBuffer());

      assertThat(scalarStuff.justI8()).isEqualTo((byte)5);
      assertThat(scalarStuff.maybeI8()).isEqualTo((byte)5);
      assertThat(scalarStuff.defaultI8()).isEqualTo((byte)5);
      assertThat(scalarStuff.justU8()).isEqualTo(6);
      assertThat(scalarStuff.maybeU8()).isEqualTo(6);
      assertThat(scalarStuff.defaultU8()).isEqualTo(6);
      assertThat(scalarStuff.justI16()).isEqualTo((short)7);
      assertThat(scalarStuff.maybeI16()).isEqualTo((short)7);
      assertThat(scalarStuff.defaultI16()).isEqualTo((short)7);
      assertThat(scalarStuff.justU16()).isEqualTo(8);
      assertThat(scalarStuff.maybeU16()).isEqualTo(8);
      assertThat(scalarStuff.defaultU16()).isEqualTo(8);
      assertThat(scalarStuff.justI32()).isEqualTo(9);
      assertThat(scalarStuff.maybeI32()).isEqualTo(9);
      assertThat(scalarStuff.defaultI32()).isEqualTo(9);
      assertThat(scalarStuff.justU32()).isEqualTo(10L);
      assertThat(scalarStuff.maybeU32()).isEqualTo(10L);
      assertThat(scalarStuff.defaultU32()).isEqualTo(10L);
      assertThat(scalarStuff.justI64()).isEqualTo(11L);
      assertThat(scalarStuff.maybeI64()).isEqualTo(11L);
      assertThat(scalarStuff.defaultI64()).isEqualTo(11L);
      assertThat(scalarStuff.justU64()).isEqualTo(12L);
      assertThat(scalarStuff.maybeU64()).isEqualTo(12L);
      assertThat(scalarStuff.defaultU64()).isEqualTo(12L);
      assertThat(scalarStuff.justF32()).isEqualTo(13.0f);
      assertThat(scalarStuff.maybeF32()).isEqualTo(13.0f);
      assertThat(scalarStuff.defaultF32()).isEqualTo(13.0f);
      assertThat(scalarStuff.justF64()).isEqualTo(14.0);
      assertThat(scalarStuff.maybeF64()).isEqualTo(14.0);
      assertThat(scalarStuff.defaultF64()).isEqualTo(14.0);
      assertThat(scalarStuff.justBool()).isTrue();
      assertThat(scalarStuff.maybeBool()).isTrue();
      assertThat(scalarStuff.defaultBool()).isTrue();
      assertThat(scalarStuff.justEnum()).isEqualTo(OptionalByte.Two);
      assertThat(scalarStuff.maybeEnum()).isEqualTo(OptionalByte.Two);
      assertThat(scalarStuff.defaultEnum()).isEqualTo(OptionalByte.Two);

      assertThat(scalarStuff.hasMaybeI8()).isTrue();
      assertThat(scalarStuff.hasMaybeI16()).isTrue();
      assertThat(scalarStuff.hasMaybeI32()).isTrue();
      assertThat(scalarStuff.hasMaybeI64()).isTrue();
      assertThat(scalarStuff.hasMaybeU8()).isTrue();
      assertThat(scalarStuff.hasMaybeU16()).isTrue();
      assertThat(scalarStuff.hasMaybeU32()).isTrue();
      assertThat(scalarStuff.hasMaybeU64()).isTrue();
      assertThat(scalarStuff.hasMaybeF32()).isTrue();
      assertThat(scalarStuff.hasMaybeF64()).isTrue();
      assertThat(scalarStuff.hasMaybeBool()).isTrue();
      assertThat(scalarStuff.hasMaybeEnum()).isTrue();
    }

    static void TestObject(MonsterT monster) {
      assertThat(monster.getHp()).isEqualTo((short) 80);
      // default
      assertThat(monster.getMana()).isEqualTo((short) 150);

      assertThat(monster.getName()).isEqualTo("MyMonster");
      assertThat(monster.getColor()).isEqualTo((Integer) Color.Blue);
      // monster.friendly() // can't access, deprecated

        Vec3T pos = monster.getPos();
      assertThat(pos.getX()).isEqualTo(1.0f);
      assertThat(pos.getY()).isEqualTo(2.0f);
      assertThat(pos.getZ()).isEqualTo(3.0f);
      assertThat(pos.getTest1()).isEqualTo(3.0);
      // issue: int != byte
      assertThat(pos.getTest2()).isEqualTo((int) Color.Green);
      TestT t = pos.getTest3();
      assertThat(t.getA()).isEqualTo((short) 5);
      assertThat(t.getB()).isEqualTo((byte) 6);

      assertThat(monster.getTest().getType()).isEqualTo((byte) Any.Monster);
      MonsterT monster2 = (MonsterT) monster.getTest().getValue();
      assertThat(monster2 != null).isTrue();
      assertThat(monster2.getName()).isEqualTo("Fred");

      int[] inv = monster.getInventory();
      assertThat(inv.length).isEqualTo(5);
      int[] expInv = {0, 1, 2, 3, 4};
        for (int i = 0; i < inv.length; i++)
          assertThat(expInv[i]).isEqualTo(inv[i]);

        TestT[] test4 = monster.getTest4();
        TestT test_0 = test4[0];
        TestT test_1 = test4[1];
      assertThat(test4.length).isEqualTo(2);
      assertThat(test_0.getA()).isEqualTo((short) 10);
      assertThat(test_0.getB()).isEqualTo((byte) 20);
      assertThat(test_1.getA()).isEqualTo((short) 30);
      assertThat(test_1.getB()).isEqualTo((byte) 40);

      String[] testarrayofstring = monster.getTestarrayofstring();
      assertThat(testarrayofstring.length).isEqualTo(2);
      assertThat(testarrayofstring[0]).isEqualTo("test1");
      assertThat(testarrayofstring[1]).isEqualTo("test2");

      MonsterT[] testarrayoftables = monster.getTestarrayoftables();
      assertThat(testarrayoftables.length).isEqualTo(0);

      MonsterT enemy = monster.getEnemy();
      assertThat(enemy != null).isTrue();
      assertThat(enemy.getName()).isEqualTo("Fred");

      int[] testnestedflatbuffer = monster.getTestnestedflatbuffer();
      assertThat(testnestedflatbuffer.length).isEqualTo(0);

      assertThat(monster.getTestempty() == null).isTrue();

      assertThat(monster.getTestbool()).isTrue();

      boolean[] testarrayofbools = monster.getTestarrayofbools();
      assertThat(testarrayofbools.length).isEqualTo(3);
      assertThat(testarrayofbools[0]).isTrue();
      assertThat(testarrayofbools[1]).isFalse();
      assertThat(testarrayofbools[2]).isTrue();

      assertThat(monster.getTestf()).isEqualTo(3.14159f);
      assertThat(monster.getTestf2()).isEqualTo(3.0f);
      assertThat(monster.getTestf3()).isEqualTo(0.0f);
      assertThat(monster.getTestf3()).isEqualTo(0.0f);

      AbilityT[] testarrayofsortedstruct = monster.getTestarrayofsortedstruct();
      assertThat(testarrayofsortedstruct.length).isEqualTo(3);
      assertThat(testarrayofsortedstruct[0].getId()).isEqualTo((long) 0);
      assertThat(testarrayofsortedstruct[1].getId()).isEqualTo((long) 1);
      assertThat(testarrayofsortedstruct[2].getId()).isEqualTo((long) 5);
      assertThat(testarrayofsortedstruct[0].getDistance()).isEqualTo((long) 45);
      assertThat(testarrayofsortedstruct[1].getDistance()).isEqualTo((long) 21);
      assertThat(testarrayofsortedstruct[2].getDistance()).isEqualTo((long) 12);

      int[] flex = monster.getFlex();
      assertThat(flex.length).isEqualTo(0);

      long[] vectorOfLongs = monster.getVectorOfLongs();
      assertThat(vectorOfLongs.length).isEqualTo(5);
      long l = 1;
        for (int i = 0; i < vectorOfLongs.length; i++) {
          assertThat(vectorOfLongs[i]).isEqualTo(l);
          l *= 100;
        }

        double[] vectorOfDoubles = monster.getVectorOfDoubles();
      assertThat(vectorOfDoubles.length).isEqualTo(3);
      assertThat(vectorOfDoubles[0]).isEqualTo(-1.7976931348623157E308);
      assertThat(vectorOfDoubles[1]).isEqualTo(0.0);
      assertThat(vectorOfDoubles[2]).isEqualTo(1.7976931348623157E308);

      assertThat(monster.getParentNamespaceTest() == null).isTrue();
      ReferrableT[] vectorOfReferrables = monster.getVectorOfReferrables();
      assertThat(vectorOfReferrables.length).isEqualTo(0);

      assertThat(monster.getSignedEnum()).isEqualTo((byte) -1);
    }

    static void TestPackUnpack(ByteBuffer bb) {
        Monster m = Monster.getRootAsMonster(bb);
        MonsterT mObject = m.unpack();
        TestObject(mObject);
        FlatBufferBuilder fbb = new FlatBufferBuilder();
        int monster = Monster.pack(fbb, mObject);
        Monster.finishMonsterBuffer(fbb, monster);
        TestBuffer(fbb.dataBuffer());

        byte[] bytes = mObject.serializeToBinary();
        MonsterT newMonsterT = MonsterT.deserializeFromBinary(bytes);
        TestObject(newMonsterT);
    }
}
