/*
 * Copyright 1997, Steven M. Robbins <steve@nyongwa.montreal.qc.ca>
 * $Id: ParseOptions.java,v 1.3 1998/05/17 04:49:55 steve Exp $
 */

package smr.JavaDeps;


import java.util.Hashtable;


/**
 * Parser for command-line options.  Inspired by C's getopt() and Paul
 * Jimenez's Commandline.  Throughout, a <em>flag</em> is taken to mean a
 * switch which is either present or not.  It does not take an argument.  An
 * <em>option</em>, on the other hand must always have an associated argument.
 *
 **/

public class ParseOptions
{
    class Option
    {
	int count;
	boolean needsArgument;
	String argument;
	
	Option( boolean na )
	{
	    count = 0;
	    needsArgument = na;
	    argument = null;
	}
    }

    // Record all the options; whether they have been seen &etc.
    private Hashtable opts;

    // Record the remaining (non-switch) arguments
    private String otherArguments[];
    
    /**
     * @param shortFlags a string containing all valid one-character flags
     * @param shortOptions a string containing all valid one-character options
     * @param arguments an array of commandline arguments to parse
     **/
    public ParseOptions( String arguments[],
			 String shortFlags, String shortOptions )
	 throws InvalidOptionException
    {
	this( arguments, shortFlags, shortOptions, null, null );
    }

    /**
     * @param shortFlags a string containing all valid one-character flags
     * @param shortOptions a string containing all valid one-character options
     * @param longFlags an array of the long flags
     * @param longOptions an array of the long options
     * @param arguments an array of commandline arguments to parse
     **/
    public ParseOptions( String arguments[],
			 String shortFlags, String shortOptions,
			 String longFlags[], String longOptions[] )
	 throws InvalidOptionException
    {
	opts = new Hashtable();

	for ( int i = 0; i < shortFlags.length(); ++i ) {
	    opts.put( new Character( shortFlags.charAt( i )),
		      new Option( false ) );
	}

	for ( int i = 0; i < shortOptions.length(); ++i ) {
	    opts.put( new Character( shortOptions.charAt( i )),
		      new Option( true ) );
	}

	if ( longFlags != null ) {
	    for ( int i = 0; i < longFlags.length; ++i ) {
		opts.put( longFlags[i], new Option( false ) );
	    }
	}

	if ( longOptions != null ) {
	    for ( int i = 0; i < longOptions.length; ++i ) {
		opts.put( longOptions[i], new Option( true ) );
	    }
	}

	int argIndex = 0;
	while ( argIndex < arguments.length ) {
	    String arg = arguments[argIndex];

	    // Check if the next item is in fact, a switch
	    //
	    if ( arg.equals( "-" ) || arg.charAt( 0 ) != '-' )
		break;
	    else if ( arg.equals( "--" ) ) {
		++argIndex;
		break;
	    }

	    if ( arg.charAt( 1 ) != '-' ) {
		// short option or group of them
		for ( int i = 1; i < arg.length(); ++i ) {
		    Option o = seenSwitch( arg.charAt( i ) );
		    if ( o.needsArgument ) {
			if ( i < arg.length() - 1 )
			    o.argument = arg.substring( i+1 );
			else if ( argIndex < arguments.length )
			    o.argument = arguments[++argIndex];
			else
			    throw new
  InvalidOptionException( "option \"" + arg.charAt( i ) +
			  "\" needs an argument" );
			break;
		    }
		}
	    } else {
		// long option or flag
		int eqInd = arg.indexOf( '=' );
		String a = ( eqInd == -1 ?
			     arg.substring( 2 ) :
			     arg.substring( 2, eqInd ) );
		Option o = seenSwitch( a );

		if ( eqInd == -1 ) {
		    if ( o.needsArgument ) {
			if ( argIndex < arguments.length )
			    o.argument = arguments[++argIndex];
			else throw new
  InvalidOptionException( "option \"" + a + "\" needs an argument" );
		    }
		} else {
		    // Change: allow any flag to specify an option if and only
		    // if it is given in the --flag=option format.
		    //		    if ( o.needsArgument )
		    o.argument = arg.substring( eqInd+1 );
			//		    else
			//			throw new
			//  InvalidOptionException( "option \"" + a + "\" illegally specified an option"
			//			  );
		}
	    }

	++argIndex;
	}

	otherArguments = new String[arguments.length - argIndex];
	for ( int i = 0; i < otherArguments.length; ++i ) {
	    otherArguments[i] = arguments[argIndex++];
	}
    }

    private Option lookupSwitch( Object s )
    {
	return (Option)opts.get( s );
    }
    
    private Option seenSwitch( Object s )
	 throws InvalidOptionException
    {
	Option o = lookupSwitch( s );

	if ( o == null )
	    throw new InvalidOptionException( "unknown option: " + s );

	o.count++;
	return o;
    }

    private Option seenSwitch( char c )
	 throws InvalidOptionException
    {
	return seenSwitch( new Character( c ) );
    }

    public boolean seenOption( String s )
    {
	Option o = lookupSwitch( s );
	return ( o != null && o.count >= 1 );
    }

    public boolean seenOption( char c )
    {
	Option o = lookupSwitch( new Character( c ) );
	return ( o != null && o.count >= 1 );
    }

    public String getOptionArgument( String s, String def )
    {
	Option o = lookupSwitch( s );
	return ( o == null || o.count == 0 ? def : o.argument );
    }

    public String getOptionArgument( char c, String def )
    {
	Option o = lookupSwitch( new Character( c ) );
	return ( o == null || o.count == 0 ? def : o.argument );
    }

    public String getOptionArgument( String s )
    {
	return getOptionArgument( s, null );
    }

    public String getOptionArgument( char c )
    {
	return getOptionArgument( c, null );
    }

    public String[] getRemainingArgs()
    {
	return otherArguments;
    }
}

