//---------------------------------------------------------------------------
// Copyright (C) 2000 Dallas Semiconductor Corporation, All Rights Reserved.
// 
// Permission is hereby granted, free of charge, to any person obtaining a 
// copy of this software and associated documentation files (the "Software"), 
// to deal in the Software without restriction, including without limitation 
// the rights to use, copy, modify, merge, publish, distribute, sublicense, 
// and/or sell copies of the Software, and to permit persons to whom the 
// Software is furnished to do so, subject to the following conditions:
// 
// The above copyright notice and this permission notice shall be included 
// in all copies or substantial portions of the Software.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS 
// OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF 
// MERCHANTABILITY,  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. 
// IN NO EVENT SHALL DALLAS SEMICONDUCTOR BE LIABLE FOR ANY CLAIM, DAMAGES 
// OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, 
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR 
// OTHER DEALINGS IN THE SOFTWARE.
// 
// Except as contained in this notice, the name of Dallas Semiconductor 
// shall not be used except as stated in the Dallas Semiconductor 
// Branding Policy. 
//---------------------------------------------------------------------------
//
//  debit.c - This utility debits money from a roving SHA iButton 
//               (DS1963S) using a coprocessor SHA iButton.
//
//  Version:   2.00
//  History:   

#include "ownet.h"   
#include "ibsha18.h"

// mode constants
enum { MD_FIND_ROVING=0, MD_READ_AUTH, MD_MONEY_VERF, MD_MONEY_CHANGE,
       MD_MONEY_WRITE, MD_READ_BACK_AUTH, MD_VERIFY_GONE };

// external function prototypes 
extern void ExitProg(char *, int);
extern int EnterNum(char *, int, long *, long, long);
extern int ReadAuth(eCertificate *, TransState *);
extern int WriteMoney(eCertificate *, TransState *);
extern int FindCoprocessor(TransState *);
extern int VerifyMoney(eCertificate *, TransState *);
extern int FindRoving(TransState *);
extern int ReadBackAuth(eCertificate *,TransState *);
extern int RovingIsPresent(TransState *ts);
extern int  owAcquire(int,char *,char *);
extern void owRelease(int,char *);
extern long msGettick(void);
extern int key_abort(void);

// local function prototypes
int PrintHeader(char *);
void output_status(int, char *);
int SetNewMoneyValue(eCertificate *,long);
int DebitRovingSHAiButton(int,long,TransState *);
void eCertPrint(eCertificate *, char *);

// globals 
long time_stamp;
// verbose output mode
int VERBOSE=0;
// file for output
FILE *fp;

//----------------------------------------------------------------------
//  This is the Main routine for debit
//
int main(short argc, char **argv)
{
   int ShowStatus=FALSE,filenum,LOOP,i,cnt,exit_code=EXIT_SUCCESS;
   char msg[200];
   int copr_port_arg;
   long amount;
   TransState ts;

   // check arguments to see if request instruction with '?' or too many
   if ((argc > 7) || (argc < 2) || ((argc > 1) && (argv[1][0] == '?' || argv[1][1] == '?')))
       ExitProg("\nusage: debit 1wire_net_port </C copr_1wire_net_port> <output_filename> \n"
                  "                            </Verbose> </Loop>\n"
              "  - debit from a roving SHA iButton using coprocessor SHA iButton\n"
              "  - 1wire_net_port - required port name\n"
              "    example: \"COM1\" (Win32 DS2480),\"/dev/cua0\" \n"
              "    (Linux DS2480),\"1\" (Win32 TMEX)\n"
              "  - </C copr_1wire_net_port> optional port name for coprocessor port\n"
              "    default is both roving and coprocessor on same port\n"
              "  - <output_filename> optional output filename\n"
              "  - </V> optional show verbose output\n"
              "  - </L> optional continuous debit loop\n"
              "  - press any key to stop\n"
              "  - version 2.00\n",EXIT_ERROR);

   VERBOSE = FALSE;
   LOOP = FALSE;

   // print the header info 
   if (!PrintHeader("debit Version 2.00"))
      exit(1);

   // check arguments 
   filenum = 0;
   copr_port_arg = 0;
   for (i = 2; i < argc; i++)
   {
      if (argv[i][0] != '/')
         filenum = i;
      else if ((argv[i][1] == 'V') || (argv[i][1] == 'v'))
         VERBOSE = TRUE;
      else if ((argv[i][1] == 'l') || (argv[i][1] == 'L'))
         LOOP = TRUE;
      else if ((argv[i][1] == 'c') || (argv[i][1] == 'C'))
      {
         // make sure there is another argument
         if ((i + 1) < argc)
         {
            copr_port_arg = i + 1;
            i++;  // increment past this arg so don't confuse with filenum
         }
      }
   }

   // open the output file  
   fp = NULL;
   if (filenum > 0)
   {
      fp = fopen(argv[filenum],"wb");
      if(fp == NULL)
      {    
         printf("ERROR, Could not open output file!\n");
         exit(1);
      }
      else
         printf("File '%s' opened to write sensor data.\n\n",
                 argv[filenum]);
   }

   // attempt to acquire the 1-Wire Net
   ts.portnum[IB_ROVING] = 0;
   if (!owAcquire(ts.portnum[IB_ROVING],argv[1],msg))
      ExitProg(msg,1);

   // check if CO_PROCESSOR port is not the same
   if (copr_port_arg)
   {
      ts.portnum[IB_COPR] = 1;
      if (!owAcquire(ts.portnum[IB_COPR],argv[copr_port_arg],msg))
         ExitProg(msg,1);
   }
   else
      ts.portnum[IB_COPR] = ts.portnum[IB_ROVING];
   
   // enter amount to debit
   amount = 50;
   if (!EnterNum("\nEnter the amount to deduct per transaction (in cents)\n", 7, &amount, 0, 9999999))
      ExitProg("Aborted entering, end program\n",EXIT_ERROR);
   else
   {
      sprintf(msg,"Amount to deduct: $%4.2f US\n\n",(float)(amount/100.0));
      output_status(LV_ALWAYS,msg);  
   }

   // attempt to find co-processor
   sprintf((char *)(&ts.copr_file[0]),"COPR");
   if (FindCoprocessor(&ts))
   {
      // loop to print ROM of device found
      cnt = sprintf(msg,"** Co-processor SHA iButton found: ");
      for (i = 7; i >= 0; i--)
         cnt += sprintf(msg + cnt,"%02X",ts.copr_rom[i]);
      cnt += sprintf(msg + cnt,"\n** Provider file: ");
      for (i = 0; i < 4; i++)
         cnt += sprintf(msg + cnt,"%c",ts.provider[i]);
      cnt += sprintf(msg + cnt,".%d\n",ts.provider[4]);
      cnt += sprintf(msg + cnt,"** Money master secret page: %d\n",ts.c_mmaster_scrt);
      cnt += sprintf(msg + cnt,"** Auth master secret page: %d\n",ts.c_amaster_scrt);
      cnt += sprintf(msg + cnt,"** Unique device (calculated) secret page: %d\n\n",ts.c_udevice_scrt);
      output_status(LV_ALWAYS,msg);
      
      // find and debit roving SHA iButtons
      exit_code = DebitRovingSHAiButton(LOOP,-amount,&ts);
   }
   else
   {
      output_status(LV_ALWAYS,"ERROR, could not find co-processor on 1-Wire Net, end program\n");
      exit_code = EXIT_ERROR;
   }

   // release the 1-Wire Net(s)
   owRelease(ts.portnum[IB_ROVING],msg);
   output_status(LV_ALWAYS,msg);
   if (copr_port_arg)
   {
      owRelease(ts.portnum[IB_COPR],msg);
      output_status(LV_ALWAYS,msg);
   }

   // close opened file
   if ((fp != NULL) && (fp != stdout))
   {
      printf("File '%s' closed.\n",
              argv[filenum]);
      fclose(fp);
   }

   // check on result of operation
   if (exit_code)
      printf("ERROR reported (%d), end program\n",exit_code);
   else
      printf("End program normally\n");

   // end the program
   exit(exit_code);

   return exit_code;
}

//--------------------------------------------------------------------------
// Set new money amount (if possible)
//
int SetNewMoneyValue(eCertificate *ec, long amount_change)
{
   long l,money;
   int i;

   // convert balance to ulong
   l = 0;
   for (i = 2; i >= 0; i--)   
   {
      l <<= 8;
      l |= ec->balance[i];
   }

   // change by 'amount_change'
   if (l + amount_change < 0)
      return FALSE;
   else
      l += amount_change;
   money = l;

   // put back in balance
   for (i = 0; i < 3; i++)
   {
      ec->balance[i] = (uchar)(l & 0xFF);
      l >>= 8;
   }

   // set transaction ID to mine
   ec->trans_id = 0x1234;

   // also increment counter
   ec->write_cycle_cnt++;

   return TRUE;
}


//--------------------------------------------------------------------------
// Print the eCertificate
//
void eCertPrint(eCertificate *ec, char *msg)
{
   ulong ul;
   int i,mcnt=0;

   // device time to transact
   time_stamp = msGettick() - time_stamp;

   mcnt = sprintf(msg,"---------------------------\n");   
   mcnt += sprintf(msg + mcnt,"| Device: ");
 
   for (i = 6; i >=0; i--)
      mcnt += sprintf(msg + mcnt,"%02X",ec->serial_num[i]);
  
   mcnt += sprintf(msg + mcnt,"\n| User data: %s\n",ec->user_data);

   // convert balance to ulong
   ul = 0;
   for (i = 2; i >= 0; i--)   
   {
      ul <<= 8;
      ul |= ec->balance[i];
   }

   mcnt += sprintf(msg + mcnt,"| Balance: $%4.2f US\n",(float)(ul/100.0)); 
   mcnt += sprintf(msg + mcnt,"| Transaction time: %d ms\n",time_stamp);
   mcnt += sprintf(msg + mcnt,"---------------------------\n");   
}


//--------------------------------------------------------------------------
// loop to find and debit sha iButtons
//
int DebitRovingSHAiButton(int LOOP, long amount, TransState *ts)
{
   eCertificate ec,vec;
   int debit_mode,check_present,mcnt;
   char msg[255],lastmsg[255];
   long tstamp;

   // start mode
   debit_mode = MD_FIND_ROVING;

   // loop to find and deduct from the money account on roving
   do
   {
      // assume need to check if device is present
      check_present = TRUE;

      // switch on current debit mode
      switch (debit_mode)
      {
         case MD_FIND_ROVING:
            // time stamp of start of search
            time_stamp = msGettick(); 
            if (FindRoving(ts))
            {
               sprintf(msg,"** Roving SHA iButton found\n");
               debit_mode = MD_READ_AUTH;
            }
            else  
               sprintf(msg,"\n** Waiting for roving SHA iButton\n");
            check_present = FALSE;
            break;

         case MD_READ_AUTH:
            if (ReadAuth(&ec,ts))
            {
               sprintf(msg,"** Read authentication complete\n");
               check_present = FALSE;
               debit_mode = MD_MONEY_VERF;
            }
            else  
               sprintf(msg,"ERROR doing read authentication\n");
            break;

         case MD_MONEY_VERF:
            // first check if correct money type 
            if (ec.code_mult != 0x8B48)
            {
               sprintf(msg,"\n** ERROR, money wrong type (code %04X)\n\n",ec.code_mult);
               check_present = FALSE;
               debit_mode = MD_VERIFY_GONE;
               break;     
            }
            // verify money valid
            if (VerifyMoney(&ec, ts))
            {
               sprintf(msg,"** Money value authenticated\n");
               check_present = FALSE;
               debit_mode = MD_MONEY_CHANGE;
            }
            else  
               sprintf(msg,"ERROR authenticating money\n");
            break;

         case MD_MONEY_CHANGE:
            // new value for the money
            if (!SetNewMoneyValue(&ec,amount))
            {
               sprintf(msg,"\n** ERROR, not enough funds available to debit\n\n");
               check_present = FALSE;
               debit_mode = MD_VERIFY_GONE;
            }
            else
            {
               sprintf(msg,"** New money value set (debit of %d)\n",-amount);
               check_present = FALSE;
               debit_mode = MD_MONEY_WRITE;
            }
            break;

         case MD_MONEY_WRITE:
            if (WriteMoney(&ec,ts))
            {
               sprintf(msg,"** New money written\n");
               check_present = FALSE;
            }
            else  
               sprintf(msg,"ERROR writing new money value\n");
            debit_mode = MD_READ_BACK_AUTH;
            break;

         case MD_READ_BACK_AUTH:
            if (ReadBackAuth(&vec,ts))
            {
               check_present = FALSE;
               mcnt = sprintf(msg,"** Read back authentication complete\n");
   
               // check if written correclty
               if (ec.write_cycle_cnt == vec.write_cycle_cnt)
               {
                  // succes so print result
                  eCertPrint(&ec,&msg[mcnt]);
                  // make sure max block ok
                  if (LOOP)
                  {
                     // if in loop mode go directly to read auth
                     // (skip finding roving)
                     debit_mode = MD_READ_AUTH; 
                     // start the time stamp now
                     time_stamp = msGettick(); 
                  }
                  else
                     debit_mode = MD_VERIFY_GONE;
               }   
               // not written so go back to that mode
               else if (ec.write_cycle_cnt == (vec.write_cycle_cnt + 1))
                  debit_mode = MD_MONEY_WRITE;
               // unknown error value
               else
                  return EXIT_ERROR;
            }
            else  
               sprintf(msg,"** Error doing read back authentication\n");
            break;

         case MD_VERIFY_GONE:
            check_present = FALSE;
            if (!RovingIsPresent(ts))
            {
               debit_mode = MD_FIND_ROVING;
               sprintf(msg,"\n** Waiting for roving SHA iButton\n");
            }
            else  
               sprintf(msg,"** Waiting for roving SHA iButton to leave\n");
            break;

         default:
            debit_mode = MD_FIND_ROVING;
            break;

      };

      // error occured so check if roving device present
      if (check_present)
      {
         // loop for up to 500ms waiting for device to return
         tstamp = msGettick() + 500;
         do { } 
         while (!RovingIsPresent(ts) && (tstamp > msGettick()) && !key_abort());
         // if not return then print message
         if (!RovingIsPresent(ts))
         {
            sprintf(msg,"\n>>>>>>>>>>   Please present roving iButton to finish transaction\n\n");
         }
      }

      // send output
      if (strcmp(lastmsg,msg) != 0)
      {
         output_status(LV_ALWAYS,msg);
         sprintf(lastmsg,"%s",msg);
      }
      else
         output_status(LV_VERBOSE,".");
   }
   while (!key_abort());

   return EXIT_SUCCESS;
}

